package com.omni.ads.container;

import com.google.inject.Inject;
import com.omni.ads.baseconfig.ApiContainer;
import com.omni.ads.baseconfig.ApiException;
import com.omni.ads.baseconfig.Constant;
import com.omni.ads.baseconfig.ResultDto;
import com.omni.ads.api.AdsGroupApi;
import com.omni.ads.exception.OmniAdsResponseException;
import com.omni.ads.model.adsconfig.HttpResponse;
import com.omni.ads.model.adscreativity.AdsAdDataResp;
import com.omni.ads.model.adscreativity.vo.HttpPage;
import com.omni.ads.model.adsgroup.*;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.ModelAttribute;

import javax.validation.groups.Default;
import java.util.List;

public class AdsGroupApiContainer extends ApiContainer {
    @Inject
    AdsGroupApi api;
    private static final Logger LOGGER = LoggerFactory.getLogger(AdsGroupApiContainer.class);
    /**
     * 添加单个广告组
     *
     * @param addForm (required)
     * @return AdsGroupUpdateInfo
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<AdsCommunalGroupVo> v3GroupAdd(@Validated(value = ValidationGroup.Add.class)
                                                    @ModelAttribute AdsGroupAddForm addForm) throws ApiException, OmniAdsResponseException {
        ResultDto<AdsCommunalGroupVo> rt = new ResultDto<>();
        try {
            AdsAddGroupResponse resp = api.adgroupsAdd(addForm);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupAdd exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }


    /**
     * 编辑单个广告组
     *
     * @param data (required)
     * @return AdsGroupUpdateInfo
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<AdsGroupUpdateInfo> v3GroupEdit(@Validated(value = ValidationGroup.Edit.class)
                                                     @ModelAttribute AdsGroupEditForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<AdsGroupUpdateInfo> rt = new ResultDto<>();
        try {
            AdsEditGroupResponse resp = api.editAdsGroup(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupEdit exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 获取单个广告组信息
     *
     * @param adGroupId (required)
     * @return AdsGroupDataResp
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<AdsGroupDataResp> v3GroupGet(@ApiParam(name = "adGroupId", value = "广告组id", required = true) Long adGroupId) throws ApiException, OmniAdsResponseException {
        ResultDto<AdsGroupDataResp> rt = new ResultDto<>();
        try {
            GetAdsGroupResponse resp = api.getAdsGroup(adGroupId);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupPage exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 批量删除广告组信息
     *
     * @param deleteForm (required)
     * @return List&lt;AdsGroupUpdateInfo&gt;
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<List<AdsGroupUpdateInfo>> v3GroupDelete(@Validated(value = ValidationGroup.Delete.class)
                                                             @ModelAttribute AdsBatchGroupDeleteForm deleteForm) throws ApiException, OmniAdsResponseException {

        ResultDto<List<AdsGroupUpdateInfo>> rt = new ResultDto<>();
        try {
            AdsDeleteGroupResponse resp = api.deleteAdsGroup(deleteForm);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupDelete exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 分页查询广告组列表
     *
     * @param data (required)
     * @return HttpResponse<HttpPage < AdsAdDataResp>>
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<HttpPage<AdsAdDataResp>> v3GroupPage(@Validated(value = {ValidationGroup.Find.class, Default.class})
                                                          @ModelAttribute AdsGroupListForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<HttpPage<AdsAdDataResp>> rt = new ResultDto<>();
        try {
            AdsGetGroupPageListResponse resp = api.getGroupList(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupPage exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 批量启停广告组
     *
     * @param data (required)
     * @return HttpResponse
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<List<AdsGroupUpdateInfo>> v3GroupStatus(@Validated(value = ValidationGroup.Status.class)
                                                             @ModelAttribute AdsBatchGroupStatusForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<List<AdsGroupUpdateInfo>> rt = new ResultDto<>();
        try {
            AdsBtachEditResponse resp = api.batchEditStatus(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupStatus exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 批量修改排期
     *
     * @param data (required)
     * @return HttpResponse
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<List<AdsGroupUpdateInfo>> v3GroupBatchSchedule(@Validated(value = ValidationGroup.Schedule.class)
                                                                    @ModelAttribute AdsBatchGroupScheduleForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<List<AdsGroupUpdateInfo>> rt = new ResultDto<>();
        try {
            AdsBtachEditResponse resp = api.batchEditSchedule(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupBatchSchedule exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 批量修改定向包
     *
     * @param data (required)
     * @return HttpResponse
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<List<AdsGroupUpdateInfo>> v3GroupBatchTarget(@Validated(value = ValidationGroup.Schedule.class)
                                                                  @ModelAttribute AdsBatchGroupTargetForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<List<AdsGroupUpdateInfo>> rt = new ResultDto<>();
        try {
            AdsBtachEditResponse resp = api.batchEditTarget(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupBatchTarget exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 批量复制广告组
     *
     * @param data (required)
     * @return HttpResponse<List < AdsGroupUpdateInfo>>
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<HttpResponse> v3GroupBatchReplicate(@Validated(value = {ValidationGroup.BatchReplicate.class})
                                           @ModelAttribute AdsBatchGroupReplicateForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<HttpResponse> rt = new ResultDto<>();
        try {
            AdsReplateResponse resp = api.batchReplicate(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupBatchReplicate exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 批量修改广告组基础出价
     *
     * @param data (required)
     * @return HttpResponse
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<List<AdsGroupUpdateInfo>> v3GroupPrice(@Validated(value = ValidationGroup.BatchPrice.class)
                                                            @ModelAttribute AdsBatchPriceForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<List<AdsGroupUpdateInfo>> rt = new ResultDto<>();
        try {
            AdsBtachEditResponse resp = api.batchEditBasePrice(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupPrice exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 批量修改广告组转化出价
     *
     * @param data (required)
     * @return HttpResponse
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<List<AdsGroupUpdateInfo>> v3GroupOcpcPrice(@Validated(value = ValidationGroup.BatchPrice.class)
                                                                @ModelAttribute AdsBatchOcpcPriceForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<List<AdsGroupUpdateInfo>> rt = new ResultDto<>();
        try {
            AdsBtachEditResponse resp = api.batchEditOcpcPrice(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupOcpcPrice exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

    /**
     * 批量修改广告组深度转化出价
     *
     * @param data (required)
     * @return HttpResponse
     * @throws ApiException If fail to call the API, e.g. server error or cannot deserialize the
     *                      response body
     */
    public ResultDto<List<AdsGroupUpdateInfo>> v3GroupDeepOcpcPrice(@Validated(value = ValidationGroup.BatchPrice.class)
                                                                    @ModelAttribute AdsBatchDeepOcpcPriceForm data) throws ApiException, OmniAdsResponseException {
        ResultDto<List<AdsGroupUpdateInfo>> rt = new ResultDto<>();
        try {
            AdsBtachEditResponse resp = api.batchEditDeepOcpcPrice(data);
            rt.setData(resp.getData());
            rt.setRet(resp.getCode());
            rt.setMsg(resp.getMessage());
            return rt;
        } catch (ApiException e) {
            LOGGER.info("v3GroupDeepOcpcPrice exception", e);
            rt.setRet(Constant.EXCUTE_FAIL);
            rt.setMsg("链接超时");
            return null;
        }
    }

}

