package com.theokanning.openai.assistants.message;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.theokanning.openai.assistants.thread.Attachment;
import com.theokanning.openai.completion.chat.ContentDeserializer;
import com.theokanning.openai.completion.chat.ContentSerializer;
import com.theokanning.openai.completion.chat.ImageContent;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.NonNull;

import java.util.List;
import java.util.Map;

/**
 * Creates a Message
 * <p>
 * https://platform.openai.com/docs/api-reference/messages/createMessage
 */
@NoArgsConstructor
@AllArgsConstructor
@Data
public class MessageRequest {
    /**
     * The role of the entity that is creating the message.  Allowed values include:
     * user: Indicates the message is sent by an actual user and should be used in most cases to represent user-generated messages.
     * assistant: Indicates the message is generated by the assistant. Use this value to insert messages from the assistant into the conversation.
     */
    @NonNull
    String role = "user";

    /**
     * The content of the message.
     */
    @NonNull
    @JsonDeserialize(using = ContentDeserializer.class)
    @JsonSerialize(using = ContentSerializer.class)
    Object content;


    /**
     * A list of files attached to the message, and the tools they should be added to.
     */
    List<Attachment> attachments;


    /**
     * Set of 16 key-value pairs that can be attached to an object.
     * This can be useful for storing additional information about the object in a structured format.
     * Keys can be a maximum of 64 characters long, and values can be a maximum of 512 characters long.
     */
    Map<String, String> metadata;

    public static MessageRequestBuilder builder() {
        return new MessageRequestBuilder();
    }

    public static class MessageRequestBuilder {
        private String role = "user";
        private Object content;
        private List<Attachment> attachments;
        private Map<String, String> metadata;
        public MessageRequestBuilder content(String content) {
            this.content = content;
            return this;
        }
        public MessageRequestBuilder content(List<ImageContent> imageContents) {
            this.content = imageContents;
            return this;
        }
        public MessageRequestBuilder role(String role) {
            this.role = role;
            return this;
        }
        public MessageRequestBuilder attachments(List<Attachment> attachments) {
            this.attachments = attachments;
            return this;
        }
        public MessageRequestBuilder metadata(Map<String, String> metadata) {
            this.metadata = metadata;
            return this;
        }
        public MessageRequest build() {
            return new MessageRequest(role, content, attachments, metadata);
        }
    }
}
