/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.util.ArrayDeque;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.JobID;
import org.apache.flink.runtime.checkpoint.CheckpointCoordinatorDeActivator;
import org.apache.flink.runtime.checkpoint.CheckpointDeclineReason;
import org.apache.flink.runtime.checkpoint.CheckpointException;
import org.apache.flink.runtime.checkpoint.CheckpointIDCounter;
import org.apache.flink.runtime.checkpoint.CheckpointProperties;
import org.apache.flink.runtime.checkpoint.CheckpointStatsTracker;
import org.apache.flink.runtime.checkpoint.CheckpointTriggerResult;
import org.apache.flink.runtime.checkpoint.CompletedCheckpoint;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStore;
import org.apache.flink.runtime.checkpoint.PendingCheckpoint;
import org.apache.flink.runtime.checkpoint.PendingCheckpointStats;
import org.apache.flink.runtime.checkpoint.RestoredCheckpointStats;
import org.apache.flink.runtime.checkpoint.StateAssignmentOperation;
import org.apache.flink.runtime.checkpoint.TaskState;
import org.apache.flink.runtime.concurrent.Future;
import org.apache.flink.runtime.concurrent.impl.FlinkCompletableFuture;
import org.apache.flink.runtime.execution.ExecutionState;
import org.apache.flink.runtime.executiongraph.Execution;
import org.apache.flink.runtime.executiongraph.ExecutionAttemptID;
import org.apache.flink.runtime.executiongraph.ExecutionJobVertex;
import org.apache.flink.runtime.executiongraph.ExecutionVertex;
import org.apache.flink.runtime.executiongraph.JobStatusListener;
import org.apache.flink.runtime.jobgraph.JobStatus;
import org.apache.flink.runtime.jobgraph.JobVertexID;
import org.apache.flink.runtime.jobgraph.tasks.ExternalizedCheckpointSettings;
import org.apache.flink.runtime.messages.checkpoint.AcknowledgeCheckpoint;
import org.apache.flink.runtime.messages.checkpoint.DeclineCheckpoint;
import org.apache.flink.runtime.state.StateObject;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CheckpointCoordinator {
    private static final Logger LOG = LoggerFactory.getLogger(CheckpointCoordinator.class);
    private static final int NUM_GHOST_CHECKPOINT_IDS = 16;
    private final Object lock = new Object();
    private final Object triggerLock = new Object();
    private final JobID job;
    private final ExecutionVertex[] tasksToTrigger;
    private final ExecutionVertex[] tasksToWaitFor;
    private final ExecutionVertex[] tasksToCommitTo;
    private final Map<Long, PendingCheckpoint> pendingCheckpoints;
    private final CompletedCheckpointStore completedCheckpointStore;
    private final String checkpointDirectory;
    private final ArrayDeque<Long> recentPendingCheckpoints;
    private final CheckpointIDCounter checkpointIdCounter;
    private final long baseInterval;
    private final long checkpointTimeout;
    private final long minPauseBetweenCheckpointsNanos;
    private final int maxConcurrentCheckpointAttempts;
    private final Timer timer;
    private JobStatusListener jobStatusListener;
    private final AtomicInteger numUnsuccessfulCheckpointsTriggers = new AtomicInteger(0);
    private ScheduledTrigger currentPeriodicTrigger;
    private long lastCheckpointCompletionNanos;
    private boolean periodicScheduling;
    private boolean triggerRequestQueued;
    private volatile boolean shutdown;
    @Nullable
    private CheckpointStatsTracker statsTracker;
    private final CheckpointProperties checkpointProperties;
    private final Executor executor;

    public CheckpointCoordinator(JobID job, long baseInterval, long checkpointTimeout, long minPauseBetweenCheckpoints, int maxConcurrentCheckpointAttempts, ExternalizedCheckpointSettings externalizeSettings, ExecutionVertex[] tasksToTrigger, ExecutionVertex[] tasksToWaitFor, ExecutionVertex[] tasksToCommitTo, CheckpointIDCounter checkpointIDCounter, CompletedCheckpointStore completedCheckpointStore, String checkpointDirectory, Executor executor) {
        Preconditions.checkArgument((baseInterval > 0L ? 1 : 0) != 0, (Object)"Checkpoint timeout must be larger than zero");
        Preconditions.checkArgument((checkpointTimeout >= 1L ? 1 : 0) != 0, (Object)"Checkpoint timeout must be larger than zero");
        Preconditions.checkArgument((minPauseBetweenCheckpoints >= 0L ? 1 : 0) != 0, (Object)"minPauseBetweenCheckpoints must be >= 0");
        Preconditions.checkArgument((maxConcurrentCheckpointAttempts >= 1 ? 1 : 0) != 0, (Object)"maxConcurrentCheckpointAttempts must be >= 1");
        if (externalizeSettings.externalizeCheckpoints() && checkpointDirectory == null) {
            throw new IllegalStateException("CheckpointConfig says to persist periodic checkpoints, but no checkpoint directory has been configured. You can configure configure one via key 'state.checkpoints.dir'.");
        }
        if (minPauseBetweenCheckpoints > 31536000000L) {
            minPauseBetweenCheckpoints = 31536000000L;
        }
        if (baseInterval < minPauseBetweenCheckpoints) {
            baseInterval = minPauseBetweenCheckpoints;
        }
        this.job = (JobID)Preconditions.checkNotNull((Object)job);
        this.baseInterval = baseInterval;
        this.checkpointTimeout = checkpointTimeout;
        this.minPauseBetweenCheckpointsNanos = minPauseBetweenCheckpoints * 1000000L;
        this.maxConcurrentCheckpointAttempts = maxConcurrentCheckpointAttempts;
        this.tasksToTrigger = (ExecutionVertex[])Preconditions.checkNotNull((Object)tasksToTrigger);
        this.tasksToWaitFor = (ExecutionVertex[])Preconditions.checkNotNull((Object)tasksToWaitFor);
        this.tasksToCommitTo = (ExecutionVertex[])Preconditions.checkNotNull((Object)tasksToCommitTo);
        this.pendingCheckpoints = new LinkedHashMap<Long, PendingCheckpoint>();
        this.checkpointIdCounter = (CheckpointIDCounter)Preconditions.checkNotNull((Object)checkpointIDCounter);
        this.completedCheckpointStore = (CompletedCheckpointStore)Preconditions.checkNotNull((Object)completedCheckpointStore);
        this.checkpointDirectory = checkpointDirectory;
        this.recentPendingCheckpoints = new ArrayDeque(16);
        this.timer = new Timer("Checkpoint Timer", true);
        if (externalizeSettings.externalizeCheckpoints()) {
            LOG.info("Persisting periodic checkpoints externally at {}.", (Object)checkpointDirectory);
            this.checkpointProperties = CheckpointProperties.forExternalizedCheckpoint(externalizeSettings.deleteOnCancellation());
        } else {
            this.checkpointProperties = CheckpointProperties.forStandardCheckpoint();
        }
        try {
            checkpointIDCounter.start();
        }
        catch (Throwable t) {
            throw new RuntimeException("Failed to start checkpoint ID counter: " + t.getMessage(), t);
        }
        this.executor = (Executor)Preconditions.checkNotNull((Object)executor);
    }

    public void setCheckpointStatsTracker(@Nullable CheckpointStatsTracker statsTracker) {
        this.statsTracker = statsTracker;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown(JobStatus jobStatus) throws Exception {
        Object object = this.lock;
        synchronized (object) {
            if (!this.shutdown) {
                this.shutdown = true;
                LOG.info("Stopping checkpoint coordinator for job " + this.job);
                this.periodicScheduling = false;
                this.triggerRequestQueued = false;
                this.timer.cancel();
                for (PendingCheckpoint pending : this.pendingCheckpoints.values()) {
                    pending.abortError(new Exception("Checkpoint Coordinator is shutting down"));
                }
                this.pendingCheckpoints.clear();
                this.completedCheckpointStore.shutdown(jobStatus);
                this.checkpointIdCounter.shutdown(jobStatus);
            }
        }
    }

    public boolean isShutdown() {
        return this.shutdown;
    }

    public Future<CompletedCheckpoint> triggerSavepoint(long timestamp, String targetDirectory) throws Exception {
        Preconditions.checkNotNull((Object)targetDirectory, (String)"Savepoint target directory");
        CheckpointProperties props = CheckpointProperties.forStandardSavepoint();
        CheckpointTriggerResult result = this.triggerCheckpoint(timestamp, props, targetDirectory, false);
        if (result.isSuccess()) {
            return result.getPendingCheckpoint().getCompletionFuture();
        }
        Exception cause = new Exception("Failed to trigger savepoint: " + result.getFailureReason().message());
        return FlinkCompletableFuture.completedExceptionally(cause);
    }

    public boolean triggerCheckpoint(long timestamp, boolean isPeriodic) {
        return this.triggerCheckpoint(timestamp, this.checkpointProperties, this.checkpointDirectory, isPeriodic).isSuccess();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    CheckpointTriggerResult triggerCheckpoint(long timestamp, CheckpointProperties props, String targetDirectory, boolean isPeriodic) {
        if (props.externalizeCheckpoint() && targetDirectory == null) {
            throw new IllegalStateException("No target directory specified to persist checkpoint to.");
        }
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                return new CheckpointTriggerResult(CheckpointDeclineReason.COORDINATOR_SHUTDOWN);
            }
            if (isPeriodic && !this.periodicScheduling) {
                return new CheckpointTriggerResult(CheckpointDeclineReason.PERIODIC_SCHEDULER_SHUTDOWN);
            }
            if (!props.forceCheckpoint()) {
                if (this.triggerRequestQueued) {
                    LOG.warn("Trying to trigger another checkpoint while one was queued already");
                    return new CheckpointTriggerResult(CheckpointDeclineReason.ALREADY_QUEUED);
                }
                if (this.pendingCheckpoints.size() >= this.maxConcurrentCheckpointAttempts) {
                    this.triggerRequestQueued = true;
                    if (this.currentPeriodicTrigger != null) {
                        this.currentPeriodicTrigger.cancel();
                        this.currentPeriodicTrigger = null;
                    }
                    return new CheckpointTriggerResult(CheckpointDeclineReason.TOO_MANY_CONCURRENT_CHECKPOINTS);
                }
                long earliestNext = this.lastCheckpointCompletionNanos + this.minPauseBetweenCheckpointsNanos;
                long durationTillNextMillis = (earliestNext - System.nanoTime()) / 1000000L;
                if (durationTillNextMillis > 0L) {
                    ScheduledTrigger trigger;
                    if (this.currentPeriodicTrigger != null) {
                        this.currentPeriodicTrigger.cancel();
                        this.currentPeriodicTrigger = null;
                    }
                    this.currentPeriodicTrigger = trigger = new ScheduledTrigger();
                    this.timer.scheduleAtFixedRate((TimerTask)trigger, durationTillNextMillis, this.baseInterval);
                    return new CheckpointTriggerResult(CheckpointDeclineReason.MINIMUM_TIME_BETWEEN_CHECKPOINTS);
                }
            }
        }
        Execution[] executions = new Execution[this.tasksToTrigger.length];
        for (int i = 0; i < this.tasksToTrigger.length; ++i) {
            Execution ee = this.tasksToTrigger[i].getCurrentExecutionAttempt();
            if (ee == null || ee.getState() != ExecutionState.RUNNING) {
                LOG.info("Checkpoint triggering task {} is not being executed at the moment. Aborting checkpoint.", (Object)this.tasksToTrigger[i].getSimpleName());
                return new CheckpointTriggerResult(CheckpointDeclineReason.NOT_ALL_REQUIRED_TASKS_RUNNING);
            }
            executions[i] = ee;
        }
        HashMap<ExecutionAttemptID, ExecutionVertex> ackTasks = new HashMap<ExecutionAttemptID, ExecutionVertex>(this.tasksToWaitFor.length);
        for (ExecutionVertex ev : this.tasksToWaitFor) {
            Execution ee = ev.getCurrentExecutionAttempt();
            if (ee == null) {
                LOG.info("Checkpoint acknowledging task {} is not being executed at the moment. Aborting checkpoint.", (Object)ev.getSimpleName());
                return new CheckpointTriggerResult(CheckpointDeclineReason.NOT_ALL_REQUIRED_TASKS_RUNNING);
            }
            ackTasks.put(ee.getAttemptId(), ev);
        }
        Object object2 = this.triggerLock;
        synchronized (object2) {
            long checkpointID;
            try {
                checkpointID = this.checkpointIdCounter.getAndIncrement();
            }
            catch (Throwable t) {
                int numUnsuccessful = this.numUnsuccessfulCheckpointsTriggers.incrementAndGet();
                LOG.warn("Failed to trigger checkpoint (" + numUnsuccessful + " consecutive failed attempts so far)", t);
                return new CheckpointTriggerResult(CheckpointDeclineReason.EXCEPTION);
            }
            final PendingCheckpoint checkpoint = new PendingCheckpoint(this.job, checkpointID, timestamp, ackTasks, props, targetDirectory, this.executor);
            if (this.statsTracker != null) {
                PendingCheckpointStats callback = this.statsTracker.reportPendingCheckpoint(checkpointID, timestamp, props);
                checkpoint.setStatsCallback(callback);
            }
            TimerTask canceller = new TimerTask(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void run() {
                    Object object = CheckpointCoordinator.this.lock;
                    synchronized (object) {
                        if (!checkpoint.isDiscarded()) {
                            LOG.info("Checkpoint " + checkpointID + " expired before completing.");
                            checkpoint.abortExpired();
                            CheckpointCoordinator.this.pendingCheckpoints.remove(checkpointID);
                            CheckpointCoordinator.this.rememberRecentCheckpointId(checkpointID);
                            CheckpointCoordinator.this.triggerQueuedRequests();
                        }
                    }
                }
            };
            try {
                Object object3 = this.lock;
                synchronized (object3) {
                    if (this.shutdown) {
                        return new CheckpointTriggerResult(CheckpointDeclineReason.COORDINATOR_SHUTDOWN);
                    }
                    if (!props.forceCheckpoint()) {
                        if (this.triggerRequestQueued) {
                            LOG.warn("Trying to trigger another checkpoint while one was queued already");
                            return new CheckpointTriggerResult(CheckpointDeclineReason.ALREADY_QUEUED);
                        }
                        if (this.pendingCheckpoints.size() >= this.maxConcurrentCheckpointAttempts) {
                            this.triggerRequestQueued = true;
                            if (this.currentPeriodicTrigger != null) {
                                this.currentPeriodicTrigger.cancel();
                                this.currentPeriodicTrigger = null;
                            }
                            return new CheckpointTriggerResult(CheckpointDeclineReason.TOO_MANY_CONCURRENT_CHECKPOINTS);
                        }
                        long earliestNext = this.lastCheckpointCompletionNanos + this.minPauseBetweenCheckpointsNanos;
                        long durationTillNextMillis = (earliestNext - System.nanoTime()) / 1000000L;
                        if (durationTillNextMillis > 0L) {
                            ScheduledTrigger trigger;
                            if (this.currentPeriodicTrigger != null) {
                                this.currentPeriodicTrigger.cancel();
                                this.currentPeriodicTrigger = null;
                            }
                            this.currentPeriodicTrigger = trigger = new ScheduledTrigger();
                            this.timer.scheduleAtFixedRate((TimerTask)trigger, durationTillNextMillis, this.baseInterval);
                            return new CheckpointTriggerResult(CheckpointDeclineReason.MINIMUM_TIME_BETWEEN_CHECKPOINTS);
                        }
                    }
                    LOG.info("Triggering checkpoint " + checkpointID + " @ " + timestamp);
                    this.pendingCheckpoints.put(checkpointID, checkpoint);
                    this.timer.schedule(canceller, this.checkpointTimeout);
                }
                for (Execution execution : executions) {
                    execution.triggerCheckpoint(checkpointID, timestamp);
                }
                this.numUnsuccessfulCheckpointsTriggers.set(0);
                return new CheckpointTriggerResult(checkpoint);
            }
            catch (Throwable t) {
                Object len$ = this.lock;
                synchronized (len$) {
                    this.pendingCheckpoints.remove(checkpointID);
                }
                int numUnsuccessful = this.numUnsuccessfulCheckpointsTriggers.incrementAndGet();
                LOG.warn("Failed to trigger checkpoint (" + numUnsuccessful + " consecutive failed attempts so far)", t);
                if (!checkpoint.isDiscarded()) {
                    checkpoint.abortError(new Exception("Failed to trigger checkpoint"));
                }
                return new CheckpointTriggerResult(CheckpointDeclineReason.EXCEPTION);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void receiveDeclineMessage(DeclineCheckpoint message) {
        if (this.shutdown || message == null) {
            return;
        }
        if (!this.job.equals((Object)message.getJob())) {
            throw new IllegalArgumentException("Received DeclineCheckpoint message for job " + message.getJob() + " while this coordinator handles job " + this.job);
        }
        long checkpointId = message.getCheckpointId();
        String reason = message.getReason() != null ? message.getReason().getMessage() : "";
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                return;
            }
            PendingCheckpoint checkpoint = this.pendingCheckpoints.get(checkpointId);
            if (checkpoint != null && !checkpoint.isDiscarded()) {
                LOG.info("Discarding checkpoint {} because of checkpoint decline from task {} : {}", new Object[]{checkpointId, message.getTaskExecutionId(), reason});
                this.pendingCheckpoints.remove(checkpointId);
                checkpoint.abortDeclined();
                this.rememberRecentCheckpointId(checkpointId);
                boolean haveMoreRecentPending = false;
                for (PendingCheckpoint p : this.pendingCheckpoints.values()) {
                    if (p.isDiscarded() || p.getCheckpointId() < checkpoint.getCheckpointId()) continue;
                    haveMoreRecentPending = true;
                    break;
                }
                if (!haveMoreRecentPending) {
                    this.triggerQueuedRequests();
                }
            } else {
                if (checkpoint != null) {
                    throw new IllegalStateException("Received message for discarded but non-removed checkpoint " + checkpointId);
                }
                if (LOG.isDebugEnabled()) {
                    if (this.recentPendingCheckpoints.contains(checkpointId)) {
                        LOG.debug("Received another decline message for now expired checkpoint attempt {} : {}", (Object)checkpointId, (Object)reason);
                    } else {
                        LOG.debug("Received decline message for unknown (too old?) checkpoint attempt {} : {}", (Object)checkpointId, (Object)reason);
                    }
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean receiveAcknowledgeMessage(AcknowledgeCheckpoint message) throws CheckpointException {
        if (this.shutdown || message == null) {
            return false;
        }
        if (!this.job.equals((Object)message.getJob())) {
            LOG.error("Received wrong AcknowledgeCheckpoint message for job {}: {}", (Object)this.job, (Object)message);
            return false;
        }
        long checkpointId = message.getCheckpointId();
        Object object = this.lock;
        synchronized (object) {
            boolean wasPendingCheckpoint;
            if (this.shutdown) {
                return false;
            }
            PendingCheckpoint checkpoint = this.pendingCheckpoints.get(checkpointId);
            if (checkpoint != null && !checkpoint.isDiscarded()) {
                switch (checkpoint.acknowledgeTask(message.getTaskExecutionId(), message.getSubtaskState(), message.getCheckpointMetaData())) {
                    case SUCCESS: {
                        LOG.debug("Received acknowledge message for checkpoint {} from task {} of job {}.", new Object[]{checkpointId, message.getTaskExecutionId(), message.getJob()});
                        if (!checkpoint.isFullyAcknowledged()) break;
                        this.completePendingCheckpoint(checkpoint);
                        break;
                    }
                    case DUPLICATE: {
                        LOG.debug("Received a duplicate acknowledge message for checkpoint {}, task {}, job {}.", new Object[]{message.getCheckpointId(), message.getTaskExecutionId(), message.getJob()});
                        break;
                    }
                    case UNKNOWN: {
                        LOG.warn("Could not acknowledge the checkpoint {} for task {} of job {}, because the task's execution attempt id was unknown. Discarding the state handle to avoid lingering state.", new Object[]{message.getCheckpointId(), message.getTaskExecutionId(), message.getJob()});
                        this.discardState(message.getJob(), message.getTaskExecutionId(), message.getCheckpointId(), message.getSubtaskState());
                        break;
                    }
                    case DISCARDED: {
                        LOG.warn("Could not acknowledge the checkpoint {} for task {} of job {}, because the pending checkpoint had been discarded. Discarding the state handle tp avoid lingering state.", new Object[]{message.getCheckpointId(), message.getTaskExecutionId(), message.getJob()});
                        this.discardState(message.getJob(), message.getTaskExecutionId(), message.getCheckpointId(), message.getSubtaskState());
                    }
                }
                return true;
            }
            if (checkpoint != null) {
                throw new IllegalStateException("Received message for discarded but non-removed checkpoint " + checkpointId);
            }
            if (this.recentPendingCheckpoints.contains(checkpointId)) {
                wasPendingCheckpoint = true;
                LOG.warn("Received late message for now expired checkpoint attempt {} from {} of job {}.", new Object[]{checkpointId, message.getTaskExecutionId(), message.getJob()});
            } else {
                LOG.debug("Received message for an unknown checkpoint {} from {} of job {}.", new Object[]{checkpointId, message.getTaskExecutionId(), message.getJob()});
                wasPendingCheckpoint = false;
            }
            this.discardState(message.getJob(), message.getTaskExecutionId(), message.getCheckpointId(), message.getSubtaskState());
            return wasPendingCheckpoint;
        }
    }

    private void completePendingCheckpoint(PendingCheckpoint pendingCheckpoint) throws CheckpointException {
        long checkpointId = pendingCheckpoint.getCheckpointId();
        CompletedCheckpoint completedCheckpoint = null;
        try {
            completedCheckpoint = pendingCheckpoint.finalizeCheckpoint();
            this.completedCheckpointStore.addCheckpoint(completedCheckpoint);
            this.rememberRecentCheckpointId(checkpointId);
            this.dropSubsumedCheckpoints(checkpointId);
        }
        catch (Exception exception) {
            if (!pendingCheckpoint.isDiscarded()) {
                pendingCheckpoint.abortError(exception);
            }
            if (completedCheckpoint != null) {
                final CompletedCheckpoint cc = completedCheckpoint;
                this.executor.execute(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            cc.discard();
                        }
                        catch (Exception nestedException) {
                            LOG.warn("Could not properly discard completed checkpoint {}.", (Object)cc.getCheckpointID(), (Object)nestedException);
                        }
                    }
                });
            }
            throw new CheckpointException("Could not complete the pending checkpoint " + checkpointId + '.', exception);
        }
        finally {
            this.pendingCheckpoints.remove(checkpointId);
            this.triggerQueuedRequests();
        }
        this.lastCheckpointCompletionNanos = System.nanoTime();
        LOG.info("Completed checkpoint {} ({} bytes in {} ms).", new Object[]{checkpointId, completedCheckpoint.getStateSize(), completedCheckpoint.getDuration()});
        if (LOG.isDebugEnabled()) {
            StringBuilder builder = new StringBuilder();
            builder.append("Checkpoint state: ");
            for (TaskState state : completedCheckpoint.getTaskStates().values()) {
                builder.append(state);
                builder.append(", ");
            }
            builder.delete(builder.length() - 2, builder.length());
            LOG.debug(builder.toString());
        }
        long timestamp = completedCheckpoint.getTimestamp();
        for (ExecutionVertex ev : this.tasksToCommitTo) {
            Execution ee = ev.getCurrentExecutionAttempt();
            if (ee == null) continue;
            ee.notifyCheckpointComplete(checkpointId, timestamp);
        }
    }

    private void rememberRecentCheckpointId(long id) {
        if (this.recentPendingCheckpoints.size() >= 16) {
            this.recentPendingCheckpoints.removeFirst();
        }
        this.recentPendingCheckpoints.addLast(id);
    }

    private void dropSubsumedCheckpoints(long checkpointId) {
        Iterator<Map.Entry<Long, PendingCheckpoint>> entries = this.pendingCheckpoints.entrySet().iterator();
        while (entries.hasNext()) {
            PendingCheckpoint p = entries.next().getValue();
            if (p.getCheckpointId() >= checkpointId || !p.canBeSubsumed()) continue;
            this.rememberRecentCheckpointId(p.getCheckpointId());
            p.abortSubsumed();
            entries.remove();
        }
    }

    private void triggerQueuedRequests() {
        if (this.triggerRequestQueued) {
            this.triggerRequestQueued = false;
            ScheduledTrigger trigger = new ScheduledTrigger();
            if (this.periodicScheduling) {
                if (this.currentPeriodicTrigger != null) {
                    this.currentPeriodicTrigger.cancel();
                }
                this.currentPeriodicTrigger = trigger;
                this.timer.scheduleAtFixedRate((TimerTask)trigger, 0L, this.baseInterval);
            } else {
                this.timer.schedule((TimerTask)trigger, 0L);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean restoreLatestCheckpointedState(Map<JobVertexID, ExecutionJobVertex> tasks, boolean errorIfNoCheckpoint, boolean allowNonRestoredState) throws Exception {
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                throw new IllegalStateException("CheckpointCoordinator is shut down");
            }
            this.completedCheckpointStore.recover();
            CompletedCheckpoint latest = this.completedCheckpointStore.getLatestCheckpoint();
            if (latest == null) {
                if (errorIfNoCheckpoint) {
                    throw new IllegalStateException("No completed checkpoint available");
                }
                return false;
            }
            LOG.info("Restoring from latest valid checkpoint: {}.", (Object)latest);
            Map<JobVertexID, TaskState> taskStates = latest.getTaskStates();
            StateAssignmentOperation stateAssignmentOperation = new StateAssignmentOperation(LOG, tasks, taskStates, allowNonRestoredState);
            stateAssignmentOperation.assignStates();
            if (this.statsTracker != null) {
                long restoreTimestamp = System.currentTimeMillis();
                RestoredCheckpointStats restored = new RestoredCheckpointStats(latest.getCheckpointID(), latest.getProperties(), restoreTimestamp, latest.getExternalPath());
                this.statsTracker.reportRestoredCheckpoint(restored);
            }
            return true;
        }
    }

    public int getNumberOfPendingCheckpoints() {
        return this.pendingCheckpoints.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getNumberOfRetainedSuccessfulCheckpoints() {
        Object object = this.lock;
        synchronized (object) {
            return this.completedCheckpointStore.getNumberOfRetainedCheckpoints();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map<Long, PendingCheckpoint> getPendingCheckpoints() {
        Object object = this.lock;
        synchronized (object) {
            return new HashMap<Long, PendingCheckpoint>(this.pendingCheckpoints);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<CompletedCheckpoint> getSuccessfulCheckpoints() throws Exception {
        Object object = this.lock;
        synchronized (object) {
            return this.completedCheckpointStore.getAllCheckpoints();
        }
    }

    public CompletedCheckpointStore getCheckpointStore() {
        return this.completedCheckpointStore;
    }

    public CheckpointIDCounter getCheckpointIdCounter() {
        return this.checkpointIdCounter;
    }

    public long getCheckpointTimeout() {
        return this.checkpointTimeout;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void startCheckpointScheduler() {
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                throw new IllegalArgumentException("Checkpoint coordinator is shut down");
            }
            this.stopCheckpointScheduler();
            this.periodicScheduling = true;
            this.currentPeriodicTrigger = new ScheduledTrigger();
            this.timer.scheduleAtFixedRate((TimerTask)this.currentPeriodicTrigger, this.baseInterval, this.baseInterval);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stopCheckpointScheduler() {
        Object object = this.lock;
        synchronized (object) {
            this.triggerRequestQueued = false;
            this.periodicScheduling = false;
            if (this.currentPeriodicTrigger != null) {
                this.currentPeriodicTrigger.cancel();
                this.currentPeriodicTrigger = null;
            }
            for (PendingCheckpoint p : this.pendingCheckpoints.values()) {
                p.abortError(new Exception("Checkpoint Coordinator is suspending."));
            }
            this.pendingCheckpoints.clear();
            this.numUnsuccessfulCheckpointsTriggers.set(0);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public JobStatusListener createActivatorDeactivator() {
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                throw new IllegalArgumentException("Checkpoint coordinator is shut down");
            }
            if (this.jobStatusListener == null) {
                this.jobStatusListener = new CheckpointCoordinatorDeActivator(this);
            }
            return this.jobStatusListener;
        }
    }

    private void discardState(final JobID jobId, final ExecutionAttemptID executionAttemptID, final long checkpointId, final StateObject stateObject) {
        if (stateObject != null) {
            this.executor.execute(new Runnable(){

                @Override
                public void run() {
                    try {
                        stateObject.discardState();
                    }
                    catch (Throwable throwable) {
                        LOG.warn("Could not properly discard state object of checkpoint {} belonging to task {} of job {}.", new Object[]{checkpointId, executionAttemptID, jobId, throwable});
                    }
                }
            });
        }
    }

    private class ScheduledTrigger
    extends TimerTask {
        private ScheduledTrigger() {
        }

        @Override
        public void run() {
            try {
                CheckpointCoordinator.this.triggerCheckpoint(System.currentTimeMillis(), true);
            }
            catch (Exception e) {
                LOG.error("Exception while triggering checkpoint.", (Throwable)e);
            }
        }
    }
}

