/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.buffer;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Collection;
import junit.framework.TestCase;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.runtime.event.AbstractEvent;
import org.apache.flink.runtime.io.network.api.EndOfPartitionEvent;
import org.apache.flink.runtime.io.network.api.serialization.EventSerializer;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferCompressor;
import org.apache.flink.runtime.io.network.buffer.BufferDecompressor;
import org.apache.flink.runtime.io.network.buffer.FreeingBufferRecycler;
import org.apache.flink.runtime.io.network.buffer.NetworkBuffer;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class BufferCompressionTest {
    private static final int BUFFER_SIZE = 0x400000;
    private static final int NUM_LONGS = 524288;
    private final boolean compressToOriginalBuffer;
    private final boolean decompressToOriginalBuffer;
    private final BufferCompressor compressor;
    private final BufferDecompressor decompressor;
    private final Buffer bufferToCompress;

    @Parameterized.Parameters(name="isDirect = {0}, codec = {1}, compressToOriginal = {2}, decompressToOriginal = {3}")
    public static Collection<Object[]> parameters() {
        return Arrays.asList({true, "LZ4", true, false}, {true, "LZ4", false, true}, {true, "LZ4", false, false}, {false, "LZ4", true, false}, {false, "LZ4", false, true}, {false, "LZ4", false, false});
    }

    public BufferCompressionTest(boolean isDirect, String compressionCodec, boolean compressToOriginalBuffer, boolean decompressToOriginalBuffer) {
        this.compressToOriginalBuffer = compressToOriginalBuffer;
        this.decompressToOriginalBuffer = decompressToOriginalBuffer;
        this.compressor = new BufferCompressor(0x400000, compressionCodec);
        this.decompressor = new BufferDecompressor(0x400000, compressionCodec);
        this.bufferToCompress = BufferCompressionTest.createBufferAndFillWithLongValues(isDirect);
    }

    @Test
    public void testCompressAndDecompressNetWorkBuffer() {
        Buffer compressedBuffer = BufferCompressionTest.compress(this.compressor, this.bufferToCompress, this.compressToOriginalBuffer);
        TestCase.assertTrue((boolean)compressedBuffer.isCompressed());
        Buffer decompressedBuffer = BufferCompressionTest.decompress(this.decompressor, compressedBuffer, this.decompressToOriginalBuffer);
        TestCase.assertFalse((boolean)decompressedBuffer.isCompressed());
        BufferCompressionTest.verifyDecompressionResult(decompressedBuffer, 0L, 524288);
    }

    @Test
    public void testCompressAndDecompressReadOnlySlicedNetworkBuffer() {
        int offset = 0x100000;
        int length = 0x200000;
        Buffer readOnlySlicedBuffer = this.bufferToCompress.readOnlySlice(offset, length);
        Buffer compressedBuffer = BufferCompressionTest.compress(this.compressor, readOnlySlicedBuffer, this.compressToOriginalBuffer);
        TestCase.assertTrue((boolean)compressedBuffer.isCompressed());
        Buffer decompressedBuffer = BufferCompressionTest.decompress(this.decompressor, compressedBuffer, this.decompressToOriginalBuffer);
        TestCase.assertFalse((boolean)decompressedBuffer.isCompressed());
        BufferCompressionTest.verifyDecompressionResult(decompressedBuffer, 131072L, 262144);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testCompressEmptyBuffer() {
        BufferCompressionTest.compress(this.compressor, this.bufferToCompress.readOnlySlice(0, 0), this.compressToOriginalBuffer);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testDecompressEmptyBuffer() {
        Buffer readOnlySlicedBuffer = this.bufferToCompress.readOnlySlice(0, 0);
        readOnlySlicedBuffer.setCompressed(true);
        BufferCompressionTest.decompress(this.decompressor, readOnlySlicedBuffer, this.decompressToOriginalBuffer);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testCompressBufferWithNonZeroReadOffset() {
        this.bufferToCompress.setReaderIndex(1);
        BufferCompressionTest.compress(this.compressor, this.bufferToCompress, this.compressToOriginalBuffer);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testDecompressBufferWithNonZeroReadOffset() {
        this.bufferToCompress.setReaderIndex(1);
        this.bufferToCompress.setCompressed(true);
        BufferCompressionTest.decompress(this.decompressor, this.bufferToCompress, this.decompressToOriginalBuffer);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testCompressNull() {
        BufferCompressionTest.compress(this.compressor, null, this.compressToOriginalBuffer);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testDecompressNull() {
        BufferCompressionTest.decompress(this.decompressor, null, this.decompressToOriginalBuffer);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testCompressCompressedBuffer() {
        this.bufferToCompress.setCompressed(true);
        BufferCompressionTest.compress(this.compressor, this.bufferToCompress, this.compressToOriginalBuffer);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testDecompressUncompressedBuffer() {
        BufferCompressionTest.decompress(this.decompressor, this.bufferToCompress, this.decompressToOriginalBuffer);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testCompressEvent() throws IOException {
        BufferCompressionTest.compress(this.compressor, EventSerializer.toBuffer((AbstractEvent)EndOfPartitionEvent.INSTANCE), this.compressToOriginalBuffer);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testDecompressEvent() throws IOException {
        BufferCompressionTest.decompress(this.decompressor, EventSerializer.toBuffer((AbstractEvent)EndOfPartitionEvent.INSTANCE), this.decompressToOriginalBuffer);
    }

    @Test
    public void testDataSizeGrowsAfterCompression() {
        int numBytes = 1;
        Buffer readOnlySlicedBuffer = this.bufferToCompress.readOnlySlice(0x200000, numBytes);
        Buffer compressedBuffer = BufferCompressionTest.compress(this.compressor, readOnlySlicedBuffer, this.compressToOriginalBuffer);
        TestCase.assertFalse((boolean)compressedBuffer.isCompressed());
        TestCase.assertEquals((Object)readOnlySlicedBuffer, (Object)compressedBuffer);
        TestCase.assertEquals((int)numBytes, (int)compressedBuffer.readableBytes());
    }

    private static Buffer createBufferAndFillWithLongValues(boolean isDirect) {
        MemorySegment segment = isDirect ? MemorySegmentFactory.allocateUnpooledSegment((int)0x400000) : MemorySegmentFactory.allocateUnpooledOffHeapMemory((int)0x400000);
        for (int i = 0; i < 524288; ++i) {
            segment.putLongLittleEndian(8 * i, (long)i);
        }
        NetworkBuffer buffer = new NetworkBuffer(segment, FreeingBufferRecycler.INSTANCE);
        buffer.setSize(0x400000);
        return buffer;
    }

    private static void verifyDecompressionResult(Buffer buffer, long start, int numLongs) {
        ByteBuffer byteBuffer = buffer.getNioBufferReadable().order(ByteOrder.LITTLE_ENDIAN);
        for (int i = 0; i < numLongs; ++i) {
            TestCase.assertEquals((long)(start + (long)i), (long)byteBuffer.getLong());
        }
    }

    private static Buffer compress(BufferCompressor compressor, Buffer buffer, boolean compressToOriginalBuffer) {
        if (compressToOriginalBuffer) {
            return compressor.compressToOriginalBuffer(buffer);
        }
        return compressor.compressToIntermediateBuffer(buffer);
    }

    private static Buffer decompress(BufferDecompressor decompressor, Buffer buffer, boolean decompressToOriginalBuffer) {
        if (decompressToOriginalBuffer) {
            return decompressor.decompressToOriginalBuffer(buffer);
        }
        return decompressor.decompressToIntermediateBuffer(buffer);
    }
}

