package com.bxm.adscounter.service.autoconfigure;

import com.bxm.warcar.cache.KeyGenerator;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

/**
 * <p> redis操作类，包括新 redis 和老 redis 两个库</p>
 *
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2018/7/26
 */
@Component
public class RedisClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(RedisClient.class);

    @Autowired
    @Qualifier("jedisPool")
    private JedisPool jedisPool;

    @Autowired
    @Qualifier("jedisPoolOld")
    private JedisPool jedisPoolOld;
    
    /**
     *  HyperLogLog 输入元素
     * @date: 2018/7/26
     * @param keyGenerator
     * @param value
     * @param dbIndex
     * @return 设置成功为1，失败为0
     */
    public Long pfadd(KeyGenerator keyGenerator, String value, Integer dbIndex){
        String key = keyGenerator.generateKey();
        if(StringUtils.isBlank(key)){
            LOGGER.error("[RedisClient] pfadd the key is null ======");
            return null;
        }
        Jedis jedis= null;
        Long count = null;
        try {
            jedis = jedisPool.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            count = jedis.pfadd(key, value);
        } catch (Exception e){
            LOGGER.error("[RedisClient] pfadd error ======",e);
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return count;
    }

    /**
     * 设置键值失效时间
     * @param keyGenerator
     * @param expireTimeInSecond 失效时间
     */
    public void expire(KeyGenerator keyGenerator, int expireTimeInSecond, Integer dbIndex){
        Preconditions.checkNotNull(keyGenerator);
        String key = keyGenerator.generateKey();
        if(expireTimeInSecond>0){
            Jedis jedis= null;
            try {
                jedis = jedisPool.getResource();
                if (null != dbIndex && dbIndex.intValue() >= 0) {
                    jedis.select(dbIndex);
                }
                jedis.expire(key,expireTimeInSecond);
            } catch (Exception e){
                LOGGER.error("[RedisClient] expire error ======",e);
            } finally {
                if(null != jedis){
                    jedis.close();
                }
            }
        }
    }

    /**
     * 为属性加上指定的增量值
     * @date: 2018/7/26
     * @param keyGenerator
     * @param field
     * @param incr_by_number
     * @param dbIndex
     * @return java.lang.Long 返回当前值
     */
    public Long hincrBy(KeyGenerator keyGenerator, String field, int incr_by_number, Integer dbIndex){
        String key = keyGenerator.generateKey();
        if(StringUtils.isBlank(key)){
            LOGGER.error("[RedisClient] hincrBy the key is null ======");
            return null;
        }
        Jedis jedis = null;
        Long result = null;
        try{
            jedis = jedisPool.getResource();
            if(null != dbIndex && dbIndex.intValue() >= 0){
                jedis.select(dbIndex);
            }
            result = jedis.hincrBy(key, field, incr_by_number);
        } catch (Exception e) {
            LOGGER.error("[RedisClient] hincrBy error ======",e);
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return result;
    }
    /**
     * 指定redis数据库，设置key的自增
     * @date: 2018/8/6
     * @param keyGenerator
     * @param dbIndex
     * @return java.lang.Long
     * @throws 
     */
    public Long incr(KeyGenerator keyGenerator, Integer dbIndex){
        return incr(keyGenerator,dbIndex,null);
    }
    /**
     * 方法描述:Redis INCR命令用于将键的整数值递增1。如果键不存在，则在执行操作之前将其设置为0。
     * 如果键包含错误类型的值或包含无法表示为整数的字符串，则会返回错误。此操作限于64位有符号整数
     * @date: 2018/7/26
     * @param keyGenerator
     * @param dbIndex
     * @return java.lang.Long
     */
    public Long incr(KeyGenerator keyGenerator, Integer dbIndex, Integer expire){
        String key = keyGenerator.generateKey();
        if(StringUtils.isBlank(key)){
            LOGGER.error("[RedisClient] incr the key is null ======");
            return null;
        }
        Jedis jedis = null;
        Long result = null;
        try {
            jedis = jedisPool.getResource();
            if(null != dbIndex && dbIndex.intValue() >= 0){
                jedis.select(dbIndex);
            }
            result = jedis.incr(key);
            if (null != expire && expire.intValue() >= 0) {
                jedis.expire(key, expire.intValue());
            }
        } catch (Exception e) {
            LOGGER.error("[RedisClient] incr error ======",e);
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return result;
    }

    /**
     * 根据 redisKey 和指定的 数据库 get value
     * @param keyGenerator
     * @param dbIndex
     * @return
     */
    public String get(KeyGenerator keyGenerator, Integer dbIndex) {
        String key = keyGenerator.generateKey();
        if(StringUtils.isBlank(key)){
            LOGGER.error("[RedisClient] get the key is null ======");
            return null;
        }
        Jedis jedis = null;
        String result = null;
        try {
            jedis = jedisPool.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            result = jedis.get(key);
        } catch (Exception e) {
            LOGGER.error("[RedisClient] get ERROR ======", e);
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return result;
    }

    /******************** up new ****** down old ****************************/

    /**
     * 用老的 reids 设置key-value值，并设置数据库和失效时间
     * @param keyGenerator
     * @param value
     * @param dbIndex
     * @param expire
     */
    public void setToOldRedis(KeyGenerator keyGenerator, String value, Integer dbIndex, Integer expire){
        String key = keyGenerator.generateKey();
        if(StringUtils.isBlank(key)){
            LOGGER.error("[RedisClient setToOldRedis] the key is null ======");
            return;
        }
        Jedis jedis = null;
        try {
            jedis = jedisPoolOld.getResource();
            if(null != dbIndex && dbIndex.intValue() >= 0){
                jedis.select(dbIndex);
            }
            jedis.set(key,value);
            if (null != expire && expire.intValue() > 0) {
                jedis.expire(key, expire.intValue());
            }
        } catch (Exception e) {
            LOGGER.error("[RedisClient setToOldRedis] error ======",e);
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }

    }

    /**
     * 从老的 redis 中获取值
     * @date: 2018/8/9
     * @param keyGenerator key
     * @param dbIndex 数据库
     * @return java.lang.String
     * @throws 
     */
    public String getFromOldRedis(KeyGenerator keyGenerator, Integer dbIndex) {
        String key = keyGenerator.generateKey();
        if(StringUtils.isBlank(key)){
            LOGGER.error("[RedisClient getFromOldRedis] the key is null ======");
            return null;
        }
        Jedis jedis = null;
        String result = null;
        try {
            jedis = jedisPoolOld.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }

            result = jedis.get(key);
        } catch (Exception e) {
            LOGGER.error("[RedisClient getFromOldRedis] ERROR ======", e);
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
        return result;
    }

    /**
     * sadd key-value 到老的 redis 库中
     * @param key
     * @param value
     * @param dbIndex 数据库
     * @param expire 有效时间
     */
    public void sAddToOldRedis(String key, String value, Integer dbIndex, Integer expire) {
        if(StringUtils.isBlank(key)){
            LOGGER.error("[RedisClient] sAddToOldRedis the key is null ======");
            return;
        }
        Jedis jedis = null;
        try {
            jedis = jedisPoolOld.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            jedis.sadd(key, value);
            if (null != expire && expire.intValue() >= 0) {
                jedis.expire(key, expire.intValue());
            }
        } catch (Exception e) {
            LOGGER.error("[RedisClient]sAddToOldRedis ERROR", e);
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
    }

    /**
     * HyperLogLog 输入元素到老的 redis
     * @date: 2018/8/10
     * @param key
     * @param value 值
     * @param dbIndex 数据库
     * @param expire 有效时间
     * @return void
     */
    public void pfaddToOldRedis(String key, String value, Integer dbIndex, Integer expire) {
        if(StringUtils.isBlank(key)){
            LOGGER.error("[RedisClient] pfaddToOldRedis the key is null ======");
            return;
        }
        Jedis jedis= null;
        try {
            jedis = jedisPoolOld.getResource();
            if (null != dbIndex && dbIndex.intValue() >= 0) {
                jedis.select(dbIndex);
            }
            jedis.pfadd(key, value);
            if (null != expire && expire.intValue() >= 0) {
                jedis.expire(key, expire);
            }
        } catch (Exception e){
            LOGGER.error("[RedisClient] pfaddToOldRedis error ======",e);
        } finally {
            if(null != jedis){
                jedis.close();
            }
        }
    }
}
