package com.bxm.adscounter.service.autoconfigure;

import com.bxm.adscounter.facade.KuaishouTrackerService;
import com.bxm.adscounter.facade.model.kuaishou.KuaishouTrackerDto;
import com.bxm.adscounter.integration.RtbConversion;
import com.bxm.adscounter.integration.RtbConversionRequest;
import com.bxm.adscounter.integration.RtbConversionResponse;
import com.bxm.adscounter.integration.aop.InvokeInterceptor;
import com.bxm.adscounter.integration.aop.RtbConversionAspect;
import com.bxm.adscounter.model.EffectEndpoint;
import com.bxm.adscounter.service.utils.UrlHelper;
import com.bxm.openlog.extension.client.Fallback;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.Production;
import com.bxm.openlog.sdk.consts.Common;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

/**
 * @author allen
 * @date 2021-11-09
 * @since 1.0
 */
@Slf4j
@Configuration
public class RtbConversionAutoConfiguration {

    private final Properties properties;
    private final KuaishouTrackerService kuaishouTrackerService;

    public RtbConversionAutoConfiguration(Properties properties, KuaishouTrackerService kuaishouTrackerService) {
        this.properties = properties;
        this.kuaishouTrackerService = kuaishouTrackerService;
    }

    @Bean
    public RtbConversionAspect rtbConversionAspect(OpenLogClient openLogClient) {
        return new RtbConversionAspect(new InvokeInterceptor() {
            @Override
            public void intercept(RtbConversion target, RtbConversionRequest request, RtbConversionResponse response) {
                EffectEndpoint endpoint = request.getEndpoint();
                if (Objects.isNull(endpoint)) {
                    return;
                }

                KeyValueMap map = new KeyValueMap();

                map.put(Common.RtbParam.P, Production.COMMON.getName());
                map.put(Common.RtbParam.MT, Common.Mt.rtbConversion.original());
                map.put(Common.RtbParam.RTB, getRtb(target));
                map.put(Common.RtbParam.BXMID, endpoint.getCountId());
                map.put(Common.RtbParam.TYPE, Optional.ofNullable(endpoint.getType()).orElse(0));
                map.put(Common.RtbParam.UID, endpoint.getUid());
                map.put(Common.RtbParam.TIME, System.currentTimeMillis());
                map.put(Common.RtbParam.ADID, endpoint.getTicketId());
                map.put(Common.RtbParam.TAGID, endpoint.getAppKey() + "-" + endpoint.getBusiness());
                map.put(Common.RtbParam.CREATEID, endpoint.getAssetsId());
                map.put(Common.RtbParam.IP, endpoint.getIp());
                map.put(Common.RtbParam.UA, UrlHelper.urlEncode(endpoint.getUserAgent()));
                map.put(Common.RtbParam.REFER, UrlHelper.urlEncode(request.getReferrer()));
                map.put(Common.RtbParam.RTB_REQUEST_URI, UrlHelper.urlEncode(request.getRequestUrl()));
                map.put(Common.RtbParam.RTB_EVENT_TYPE, request.getEventType());
                map.put(Common.RtbParam.RTB_REQ_BODY, UrlHelper.urlEncode(request.getRequestBody()));
                map.put(Common.RtbParam.RTB_RES_BODY, UrlHelper.urlEncode(response.getBody()));
                map.put(Common.RtbParam.RTB_RES_MSG, UrlHelper.urlEncode(response.getMsg()));

                this.fillDevice(target, request, map);

                String openLogRequestUri = map.createOpenLogRequestUri(properties.getOpenLogRequestDomain(), false);
                openLogClient.asyncRequest(openLogRequestUri, new Consumer<Fallback>() {
                    @Override
                    public void accept(Fallback fallback) {
                        log.warn("OpenLog request fail: {}", fallback.getUrl());
                    }
                });
            }

            private String getRtb(RtbConversion target) {
                if (target.getType() == RtbConversion.Type.Kuaishou) {
                    return Common.RtbType.Kuaishou.getId();
                } else if (target.getType() == RtbConversion.Type.Lagesi) {
                    return Common.RtbType.Lagesi.getId();
                }
                return target.getType().name();
            }

            private void fillDevice(RtbConversion target, RtbConversionRequest request, KeyValueMap map) {
                if (target.getType() == RtbConversion.Type.Kuaishou) {
                    String referrer = request.getReferrer();
                    if (org.apache.commons.lang.StringUtils.isBlank(referrer)) {
                        return;
                    }
                    String url = UrlHelper.urlDecode(referrer);
                    String cbvalue = UrlHelper.getFirstValueOfParamName(url, "callback");
                    if (org.apache.commons.lang.StringUtils.isBlank(cbvalue)) {
                        return;
                    }
                    KuaishouTrackerDto trackerDto = kuaishouTrackerService.getTrackerDto(cbvalue);
                    if (Objects.isNull(trackerDto)) {
                        return;
                    }
                    if (org.apache.commons.lang.StringUtils.isNotBlank(trackerDto.getImei2())) {
                        map.put(Common.RtbParam.IMEI_MD5, trackerDto.getImei2());
                    }
                    if (org.apache.commons.lang.StringUtils.isNotBlank(trackerDto.getIdfa2())) {
                        map.put(Common.RtbParam.IDFA_MD5, trackerDto.getIdfa2());
                    }
                    if (org.apache.commons.lang.StringUtils.isNotBlank(trackerDto.getAndroidid2())) {
                        map.put(Common.RtbParam.ANDROIDID_MD5, trackerDto.getAndroidid2());
                    }
                    if (StringUtils.isNotBlank(trackerDto.getOaid())) {
                        map.put(Common.RtbParam.OAID, trackerDto.getOaid());
                    }
                    if (StringUtils.isNotBlank(trackerDto.getOaid2())) {
                        map.put(Common.RtbParam.OAID_MD5, trackerDto.getOaid2());
                    }
                }
            }
        });
    }
}
