package com.bxm.adscounter.service.impl;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import com.bxm.adscounter.facade.model.SspUserData;
import com.bxm.adscounter.integration.UserMappingIntegration;
import com.bxm.adscounter.service.SspUserService;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

/**
 * ssp用户信息<br/>
 *
 * @author kerry.jiang
 * @date 2021/9/9 18:31
 */
@Slf4j
@Service
public class SspUserServiceImpl implements SspUserService {

    private static final int EXPIRE_TIME = 7 * 24 * 60 * 60;

    private final Counter counter;
    private final UserMappingIntegration userMappingIntegration;

    public SspUserServiceImpl(@Qualifier("jedisCounter") Counter counter,
                              UserMappingIntegration userMappingIntegration) {
        this.counter = counter;
        this.userMappingIntegration = userMappingIntegration;
    }

    @Override
    public void plusActivityJoin(String appId, String positionId, String uid) {
        String cuid = userMappingIntegration.getByCuid(appId, uid, Boolean.TRUE);
        if (StringUtils.isBlank(cuid)) {
            log.warn("cuid is empty, appId={},positionId={},uid={}",
                    appId, positionId, uid);
            return;
        }
        // TODO 有可能因单个 hash 过大引起 Redis 分区数据倾斜而不稳定。
        counter.hincrementAndGet(getActivityJoinKeyGenerator(appId), cuid, EXPIRE_TIME);
        if(StringUtils.isNotBlank(positionId)){
            counter.hincrementAndGet(getActivityJoinPositionKeyGenerator(appId, positionId), cuid, EXPIRE_TIME);
        }
    }

    @Override
    public List<SspUserData> getActivityJoinByAppId(String appId, String[] cuids, String date) {
        if (ArrayUtils.isEmpty(cuids) || cuids.length > 999) {
            throw new IllegalArgumentException("cuids");
        }
        Object original = counter.getClientOriginal();
        if (!(original instanceof JedisPool)) {
            return Collections.emptyList();
        }
        JedisPool jedisPool = (JedisPool) original;
        try (Jedis jedis = jedisPool.getResource()) {
            Map<String, Response<String>> result = Maps.newHashMap();
            Pipeline pipelined = jedis.pipelined();
            for (String cuid : cuids) {
                KeyGenerator k = getActivityJoinKeyGenerator(date, appId);
                result.put(cuid, pipelined.hget(k.generateKey(), cuid));
            }
            pipelined.syncAndReturnAll();

            if (MapUtils.isNotEmpty(result)) {
                List<SspUserData> rs = Lists.newArrayListWithCapacity(result.size());
                for (Map.Entry<String, Response<String>> entry : result.entrySet()) {
                    String cuid = entry.getKey();
                    Response<String> redisRes = entry.getValue();
                    long times = NumberUtils.toLong(redisRes.get());
                    rs.add(SspUserData.builder()
                            .cuid(cuid)
                            .isActivityJoin(times > 0)
                            .build());
                }
                return rs;
            }
        }
        return null;
    }

    @Override
    public List<SspUserData> getActivityJoinByAppId(String appId, String positionId, String[] cuids, String date) {
        if (ArrayUtils.isEmpty(cuids) || cuids.length > 999) {
            throw new IllegalArgumentException("cuids");
        }
        Object original = counter.getClientOriginal();
        if (!(original instanceof JedisPool)) {
            return Collections.emptyList();
        }
        JedisPool jedisPool = (JedisPool) original;
        try (Jedis jedis = jedisPool.getResource()) {
            Map<String, Response<String>> result = Maps.newHashMap();
            Pipeline pipelined = jedis.pipelined();
            for (String cuid : cuids) {
                KeyGenerator k = getActivityJoinPositionKeyGenerator(date, appId, positionId);
                result.put(cuid, pipelined.hget(k.generateKey(), cuid));
            }
            pipelined.syncAndReturnAll();

            if (MapUtils.isNotEmpty(result)) {
                List<SspUserData> rs = Lists.newArrayListWithCapacity(result.size());
                for (Map.Entry<String, Response<String>> entry : result.entrySet()) {
                    String cuid = entry.getKey();
                    Response<String> redisRes = entry.getValue();
                    long times = NumberUtils.toLong(redisRes.get());
                    rs.add(SspUserData.builder()
                            .cuid(cuid)
                            .isActivityJoin(times > 0)
                            .build());
                }
                return rs;
            }
        }
        return null;
    }

    private static KeyGenerator getActivityJoinKeyGenerator(String appId) {
        return getActivityJoinKeyGenerator(DateHelper.getDate(), appId);
    }

    private static KeyGenerator getActivityJoinKeyGenerator(String date, String appId) {
        return () -> KeyBuilder.build("ADX", "COUNTER", "ACTIVITY_JOIN", "SSP", date, appId);
    }

    private static KeyGenerator getActivityJoinPositionKeyGenerator(String appId, String positionId) {
        return getActivityJoinPositionKeyGenerator(DateHelper.getDate(), appId, positionId);
    }

    private static KeyGenerator getActivityJoinPositionKeyGenerator(String date, String appId, String positionId) {
        return () -> KeyBuilder.build("ADX", "COUNTER", "ACTIVITY_JOIN", "SSP", "POSITION", date, appId, positionId);
    }
}
