package com.bxm.adscounter.service.listeners.general.activity.show;

import com.bxm.adscounter.model.GeneralEndpoint;
import com.bxm.adscounter.service.events.ActivityShowEvent;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.*;
import com.google.common.collect.Maps;
import com.taobao.api.ApiException;
import com.taobao.api.DefaultTaobaoClient;
import com.taobao.api.TaobaoClient;
import com.taobao.api.request.TmcMessageProduceRequest;
import com.taobao.api.response.TmcMessageProduceResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * @author allen
 * @date 2021-12-09
 * @since 1.0
 */
@Slf4j
@Configuration
public class TaobaoReportingActivityShowEventListener implements EventListener<ActivityShowEvent> {

    private static final String TOPIC = "mmc_pos_AdCatUserBack";
    private static final String APP_KEY = "33350421";
    private static final String APP_SECRET = "ae908415819249956e6d2a17d5539c32";

    private final Counter counter;
    private final Updater updater;
    private final TaobaoClient taobaoClient;

    public TaobaoReportingActivityShowEventListener(Counter counter, Updater updater) {
        this.counter = counter;
        this.updater = updater;
        this.taobaoClient = new DefaultTaobaoClient("http://gw.api.taobao.com/router/rest", APP_KEY, APP_SECRET,
                "json", 1000, 2000);
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(ActivityShowEvent event) {
        GeneralEndpoint endpoint = event.getEndpoint();
        String referrer = endpoint.getReferrer();

        if (StringUtils.isBlank(referrer)) {
            return;
        }

        String uid = endpoint.getUid();

        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();

        MultiValueMap<String, String> params = build.getQueryParams();
        String storeId = params.getFirst("storeId");
        String taskId = params.getFirst("taskId");

        if (StringUtils.isBlank(storeId) || StringUtils.isBlank(taskId)) {
            return;
        }

        if (isDuplicateOnToday(uid)) {
            return;
        }

        this.report(uid, storeId, taskId);

        this.storeReferrer(uid, referrer);
    }

    private void report(String uid, String storeId, String taskId) {
        Map<String, Object> param = Maps.newHashMap();
        param.put("message_id", UUIDHelper.generate());
        param.put("message_type", "exposure");
        param.put("store_id", NumberUtils.toLong(storeId));
        param.put("uid", uid);
        param.put("task_id", taskId);
        param.put("timestamp", LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
        param.put("time", DateHelper.format("yyyy-MM-dd"));
        String content = JsonHelper.convert(param);

        TmcMessageProduceRequest request = new TmcMessageProduceRequest();
        request.setContent(content);
        request.setTopic(TOPIC);
        request.setTargetGroup("default");

        try {
            TmcMessageProduceResponse response = taobaoClient.execute(request);
            if (Objects.isNull(response) || !Optional.ofNullable(response.getIsSuccess()).orElse(Boolean.FALSE)) {
                log.info("Taobao report is fail! {}", content);
            }
        } catch (ApiException e) {
            log.error("execute: ", e);
        }
    }

    private boolean isDuplicateOnToday(String uid) {
        KeyGenerator key = () -> KeyBuilder.build("tmp", "tb", "rep", "show", DateHelper.getDate(), LocalTime.now().getHour());
        long exp = DateHelper.getRemainSecondsOfToday() + RandomUtils.nextInt(0, 600);
        return counter.hincrementAndGet(key, uid, TypeHelper.castToInt(exp)) > 1;
    }

    private void storeReferrer(String uid, String referrer) {
        updater.updateWithSelector(() -> KeyBuilder.build("tmp", "tb", "refer", uid), referrer, 8 * 24 * 60 * 60, 6);
    }
}
