package com.bxm.adscounter.service.listeners.general.ticket.show;

import com.bxm.adscounter.model.GeneralEndpoint;
import com.bxm.adscounter.service.events.TicketShowEvent;
import com.bxm.adsprod.facade.ticket.TicketAssets;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.warcar.cache.impls.redis.JedisCounter;
import com.bxm.warcar.cache.impls.redis.JedisFetcher;
import com.bxm.warcar.cache.impls.redis.JedisUpdater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.DateHelper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;


/**
 * @author: zm
 * @date: 2019-09-02 11:02
 * @Description: fm出券算法广告券统计相关(全局数据存储)
 */
@Service
public class FmGeneralTicketShowCountEventListener implements EventListener<TicketShowEvent> {


    private static final Logger logger = LoggerFactory.getLogger(FmGeneralTicketShowCountEventListener.class);


    @Autowired
    @Qualifier("jedisFetcher")
    private com.bxm.warcar.cache.Fetcher fetcher;

    @Autowired
    @Qualifier("jedisFetcherForFM")
    private JedisFetcher jedisFetcherForFM;

    @Autowired
    @Qualifier("jedisUpdaterForFM")
    private JedisUpdater jedisUpdaterForFM;


    @Autowired
    @Qualifier("jedisCounterForFM")
    private JedisCounter jedisCounterForFM;


    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(TicketShowEvent event) {
        Date currentDate = new Date();
        GeneralEndpoint endpoint = event.getEndpoint();
        if (endpoint == null || StringUtils.isBlank(endpoint.getUid())) {
            return;
        }
        //设置全局数据
        saveGloabelDataFromFm(endpoint, currentDate);
    }

    /**
     * 设置全局数据
     *
     * @param endpoint
     * @param currentDate
     */
    private void saveGloabelDataFromFm(GeneralEndpoint endpoint, Date currentDate) {
        //存储曝光时间 3d
        jedisUpdaterForFM.hupdate(TicketKeyGenerator.TicketFm.getLaterUser(endpoint.getUid()),
                TicketKeyGenerator.TicketFm.getPrevShowDate().generateKey(), currentDate,
                (int) TimeUnit.DAYS.toSeconds(3));

        //存储活动曝光时间
        if (endpoint.getActivityId() != null) {
            jedisUpdaterForFM.hupdate(TicketKeyGenerator.TicketFm.getLaterUser(endpoint.getUid()),
                    TicketKeyGenerator.TicketFm.getPrevActivityShowDate(endpoint.getActivityId()).generateKey(),
                    currentDate, (int) TimeUnit.DAYS.toSeconds(3));
        }

        //存本素材id与素材特征（3日）
        if(StringUtils.isNotBlank(endpoint.getAssetsId())){
            List<TicketAssets>  assetsList = fetcher.hfetchList(TicketKeyGenerator.getTicketAssets(), String.valueOf(endpoint.getTicketId()), null, TicketAssets.class);
            if(!CollectionUtils.isEmpty(assetsList)){
                for(TicketAssets ticketAssets:assetsList){
                    if(StringUtils.equalsIgnoreCase(endpoint.getAssetsId(),ticketAssets.getId().toString())){
                        jedisUpdaterForFM.hupdate(TicketKeyGenerator.TicketFm.getLaterUser(endpoint.getUid()),
                                TicketKeyGenerator.TicketFm.getRecordAssetId().generateKey(),ticketAssets.getId()+"-"+ticketAssets.getIsShieldPopup() , (int) TimeUnit.DAYS.toSeconds(3));
                    }
                    break;
                }
            }
        }


        String currentDay = DateHelper.format(currentDate, DateHelper.PATTERN_STR10);
        long userActivityCount = 0L;
        long userTicketCount = 0L;

        long userAssetIdCount = 0L;

        List<Date> prev2Day = getPrev2Day(currentDay);
        List<String> removeFields = new ArrayList<>(6);

        // 获取前2天的数据
        for (Date day : prev2Day) {

            Long activityCount = null;
            if (endpoint.getActivityId() != null) {
                activityCount = jedisFetcherForFM.hfetch(TicketKeyGenerator.TicketFm.getSeqLatelyUser(endpoint.getUid()),
                        TicketKeyGenerator.TicketFm.getSeqCount(day, endpoint.getActivityId()).generateKey(), Long.class);
                if (activityCount != null) {
                    removeFields.add(TicketKeyGenerator.TicketFm.getSeqCount(day, endpoint.getActivityId()).generateKey());
                }
            }

            Long currentDayTicketCount = jedisFetcherForFM.hfetch(TicketKeyGenerator.TicketFm.getSeqLatelyUser(endpoint.getUid()),
                    TicketKeyGenerator.TicketFm.getSeqCountNotTicketId(day).generateKey(), Long.class);
            if (currentDayTicketCount != null) {
                removeFields.add(TicketKeyGenerator.TicketFm.getSeqCountNotTicketId(day).generateKey());
            }
            Long assetIdCount =null;
            if(StringUtils.isNotBlank(endpoint.getAssetsId())){
                assetIdCount = jedisFetcherForFM.hfetch(TicketKeyGenerator.TicketFm.getSeqLatelyUser(endpoint.getUid()),
                        TicketKeyGenerator.TicketFm.getSeqAssetIdCount(day,Long.valueOf(endpoint.getAssetsId())).generateKey(), Long.class);
                if (assetIdCount != null) {
                    removeFields.add(TicketKeyGenerator.TicketFm.getSeqAssetIdCount(day,Long.valueOf(endpoint.getAssetsId())).generateKey());
                }
            }



            if (activityCount != null) {
                userActivityCount += activityCount;
            }
            if (currentDayTicketCount != null) {
                userTicketCount += currentDayTicketCount;
            }
            if (assetIdCount != null) {
                userAssetIdCount += assetIdCount ;
            }
        }

        //批量删除
        if (!CollectionUtils.isEmpty(removeFields)) {
            jedisUpdaterForFM.hremove(TicketKeyGenerator.TicketFm.getSeqLatelyUser(endpoint.getUid()),
                    removeFields.toArray(new String[0]));
        }

        //设置用户全局发券订单次序 针对uid 串行化 若uid并行化,可能存在多计算前2天的数据
        jedisCounterForFM.hincrementByAndGet(TicketKeyGenerator.TicketFm.getSeqLatelyUser(endpoint.getUid()),
                TicketKeyGenerator.TicketFm.getSeqCountNotTicketId(currentDate).generateKey(), userTicketCount + 1, (int) TimeUnit.DAYS.toSeconds(3));

        if (endpoint.getActivityId() != null) {
            //设置用户全局活动订单次序
            jedisCounterForFM.hincrementByAndGet(TicketKeyGenerator.TicketFm.getSeqLatelyUser(endpoint.getUid()),
                    TicketKeyGenerator.TicketFm.getSeqCount(currentDate, endpoint.getActivityId()).generateKey(), userActivityCount + 1, (int) TimeUnit.DAYS.toSeconds(3));
        }

        if (StringUtils.isNotBlank(endpoint.getAssetsId())) {
            //设置用户全局素材次序
            jedisCounterForFM.hincrementByAndGet(TicketKeyGenerator.TicketFm.getSeqLatelyUser(endpoint.getUid()),
                    TicketKeyGenerator.TicketFm.getSeqAssetIdCount(currentDate,Long.valueOf(endpoint.getAssetsId())).generateKey(), userAssetIdCount + 1, (int) TimeUnit.DAYS.toSeconds(3));
        }

        //设置用户每天每张券出券次数
        jedisCounterForFM.hincrementByAndGet(TicketKeyGenerator.TicketFm.getUserEveryTicketCount(endpoint.getUid()),
                String.valueOf(endpoint.getTicketId()), 1, (int) DateHelper.getRemainSecondsOfToday());
    }


    /**
     * 获取前2天
     *
     * @return
     */
    private List<Date> getPrev2Day(String currentDay) {
        String preDate = null;
        String preDate2 = null;
        try {
            preDate = DateHelper.getPreDate(currentDay);
            preDate2 = DateHelper.getPreDate(preDate);
        } catch (ParseException e) {
            logger.error("get prev3 day error ", e);
        }
        return Stream.of(preDate, preDate2).filter(e -> e != null).map(e -> DateHelper.parse(e, DateHelper.PATTERN_STR10)).collect(Collectors.toList());
    }

}
