package com.bxm.adscounter.service.openlog.common.feedback;

import com.bxm.adscounter.model.RtbAdvertiser;
import com.bxm.adscounter.model.RtbMedia;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.listener.eventbus.OpenLogEvent;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 根据openlog参数，分发到指定的媒体回传接口
 * @author tangx
 * @date 2022/7/20 15:46
 */
@Configuration
@Slf4j
public class ConversionFeedbackDispatcher {

    private final ConversionFeedbackFactory conversionFeedbackFactory;

    public ConversionFeedbackDispatcher(ConversionFeedbackFactory conversionFeedbackFactory) {
        this.conversionFeedbackFactory = conversionFeedbackFactory;
    }

    public void dispatch(OpenLogEvent event) {
        KeyValueMap feedback = event.getLog();
        String bxmmedia = feedback.getFirst(Common.Param.BXMMEDIA);
        String bxmadvertiser = feedback.getFirst(Common.Param.BXMADVERTISER);
        if (StringUtils.isBlank(bxmmedia)) {
            log.warn("[ConversionFeedbackDispatcher] Missing param 'bxmmedia'.");
            return;
        }
        if (StringUtils.isBlank(bxmadvertiser)) {
            log.warn("[ConversionFeedbackDispatcher] Missing param 'bxmadvertiser'.");
            return;
        }

        RtbMedia rtbMedia = RtbMedia.getByType(bxmmedia);
        if (Objects.isNull(rtbMedia)) {
            log.warn("[ConversionFeedbackDispatcher] Illegal bxmmedia [{}]", bxmmedia);
            return;
        }
        RtbAdvertiser rtbAdvertiser = RtbAdvertiser.getByType(bxmadvertiser);
        if (Objects.isNull(rtbAdvertiser)) {
            log.warn("[ConversionFeedbackDispatcher] Illegal bxmadvertiser [{}]", bxmadvertiser);
            return;
        }

        ConversionFeedback conversionFeedback = conversionFeedbackFactory.get(rtbMedia);
        if (Objects.isNull(conversionFeedback)) {
            log.warn("[ConversionFeedbackDispatcher] Cannot found target ConversionFeedback, rtbMedia: [{}]", rtbMedia);
            return;
        }

        String clickId = feedback.getFirst(Common.Param.CLICK_ID);
        if (StringUtils.isBlank(clickId)) {
            log.warn("[ConversionFeedbackDispatcher] Missing param 'click_id'.");
            return;
        }

        String convDef = feedback.getFirst(Common.Param.CONV_DEF);
        if (StringUtils.isBlank(convDef)) {
            log.warn("[ConversionFeedbackDispatcher] Missing param 'conv_def'.");
            return;
        }

        String transformType = feedback.getFirst(rtbAdvertiser.getTransferType());
        String targetType = getTargetType(transformType, convDef);
        if (StringUtils.isBlank(targetType)) {
            log.warn("[ConversionFeedbackDispatcher] Cannot found target type [{}] . convDef: {}", transformType, convDef);
            return;
        }

        if (doIntercept(feedback)) {
            return;
        }

        conversionFeedback.feedback(feedback, clickId, targetType);
    }

    /**
     * 概率回传拦截
     */
    private boolean doIntercept(KeyValueMap m) {
        String convRatio = m.getFirst(Common.Param.CONV_RATIO);
        double maxRate = 100d;
        if (StringUtils.isNotBlank(convRatio)) {
            double ratio = Double.parseDouble(convRatio);
            if (ratio < maxRate) {
                double random = RandomUtils.nextDouble(0, maxRate);
                if (random > ratio) {
                    log.info("[ReporterDispatcher] Conversion feedback is intercepted by convRatio[{}]", convRatio);
                    return true;
                }
            }
        }
        return false;
    }


    /**
     * 获取目标的转化类型
     * @return
     */
    String getTargetType(String transformType, String convDef) {
        Map<String, String> conv = new HashMap<>();
        String[] strings = StringUtils.split(convDef, ",");
        for (String string : strings) {
            String[] arr = StringUtils.split(string, "-");
            if (arr.length < 2) {
                continue;
            }
            String src = arr[0];
            String dest = arr[1];
            conv.put(src, dest);
        }

        return conv.get(transformType);
    }

}
