package com.bxm.adscounter.service.openlog.common.report;

import com.bxm.adscounter.model.RtbAdvertiser;
import com.bxm.adscounter.model.RtbMedia;
import com.bxm.adscounter.service.autoconfigure.Properties;
import com.bxm.adscounter.service.openlog.common.feedback.ConversionFeedback;
import com.bxm.adscounter.service.openlog.common.feedback.ConversionFeedbackFactory;
import com.bxm.adscounter.service.utils.UrlHelper;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.Production;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.listener.eventbus.OpenLogEvent;
import com.bxm.openlog.sdk.params.ProductionCommonParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.annotation.Configuration;

import java.util.Objects;

/**
 * 根据openlog参数，分发到指定的广告主上报接口
 * @author tangx
 * @date 2022/7/20 15:46
 */
@Configuration
@Slf4j
public class ReporterDispatcher {

    private final ConversionFeedbackFactory conversionFeedbackFactory;
    private final ReporterFactory reporterFactory;
    private final Properties properties;

    public ReporterDispatcher(ConversionFeedbackFactory conversionFeedbackFactory, ReporterFactory reporterFactory, Properties properties) {
        this.conversionFeedbackFactory = conversionFeedbackFactory;
        this.reporterFactory = reporterFactory;
        this.properties = properties;
    }

    public void dispatch(OpenLogEvent event) {
        KeyValueMap clickTrackerLog = event.getLog();
        String bxmmedia = clickTrackerLog.getFirst(Common.Param.BXMMEDIA);
        String bxmadvertiser = clickTrackerLog.getFirst(Common.Param.BXMADVERTISER);

        if (StringUtils.isBlank(bxmmedia)) {
            log.warn("[ReporterDispatcher] Missing param '{}'.", Common.Param.BXMMEDIA);
            return;
        }
        if (StringUtils.isBlank(bxmadvertiser)) {
            log.warn("[ReporterDispatcher] Missing param '{}'.", Common.Param.BXMADVERTISER);
            return;
        }

        RtbMedia rtbMedia = RtbMedia.getByType(bxmmedia);
        RtbAdvertiser rtbAdvertiser = RtbAdvertiser.getByType(bxmadvertiser);

        if (Objects.isNull(rtbMedia)) {
            log.warn("[ReporterDispatcher] Illegal rtbMedia [{}]", bxmmedia);
            return;
        }
        if (Objects.isNull(rtbAdvertiser)) {
            log.warn("[ReporterDispatcher] Illegal rtbAdvertiser [{}]", bxmadvertiser);
            return;
        }

        ConversionFeedback rtbInstance = conversionFeedbackFactory.get(rtbMedia);
        if (Objects.isNull(rtbInstance)) {
            return;
        }
        rtbInstance.update(clickTrackerLog);

        String extParams = rtbInstance.createExtParams(clickTrackerLog);

        Reporter reporter = reporterFactory.get(rtbAdvertiser);

        if (Objects.isNull(reporter)) {
            log.warn("[ReporterDispatcher] Cannot found target reporter, rtbAdvertiser: {}", rtbAdvertiser);
            return;
        }

        String feedbackUrl = createBxmFeedbackUrl(clickTrackerLog, extParams, rtbMedia);

        reporter.report(clickTrackerLog, feedbackUrl);
    }

    public String createBxmFeedbackUrl(KeyValueMap clickTrackerLog, String extParams, RtbMedia rtbMedia) {
        KeyValueMap map = new KeyValueMap();
        map.setProduction(Production.COMMON);
        map.setMt(Common.Mt.oCpxFeedbackForCommon.original());
        map.putIfNotBlank(Common.Param.BXMMEDIA, clickTrackerLog.getFirst(Common.Param.BXMMEDIA));
        map.putIfNotBlank(Common.Param.BXMADVERTISER, clickTrackerLog.getFirst(Common.Param.BXMADVERTISER));
        map.putIfNotBlank(Common.Param.BXMEXTRA, clickTrackerLog.getFirst(Common.Param.BXMEXTRA));

        String clickId = clickTrackerLog.getFirst(Common.Param.CLICK_ID);

        // 广点通click_id做base64加密
        if (RtbMedia.GUANGDIANTONG.equals(rtbMedia)) {
            if (StringUtils.isNotBlank(clickId)) {
                clickId = UrlHelper.urlEncode(Base64.encodeBase64String(clickId.getBytes()));
            }
        }

        if (RtbMedia.XIAOMI.equals(rtbMedia)) {
            String imei = clickTrackerLog.getFirst(ProductionCommonParam.IMEI);
            String oaid = clickTrackerLog.getFirst(ProductionCommonParam.OAID);
            map.putIfNotBlank("x_imei", imei);
            map.putIfNotBlank("x_oaid", oaid);
        }

        map.putIfNotBlank(Common.Param.CLICK_ID, clickId);

        map.putIfNotBlank(Common.Param.CONV_DEF, clickTrackerLog.getFirst(Common.Param.CONV_DEF));
        map.putIfNotBlank(Common.Param.CONV_RATIO, clickTrackerLog.getFirst(Common.Param.CONV_RATIO));
        map.putIfNotBlank(Reporter.NAME_EXT_PARAMS, extParams);
        return map.createOpenLogRequestUri(properties.getOpenLogRequestOutDomain());
    }

}
