package com.bxm.adscounter.service.openlog.common.report.impl;

import com.bxm.adscounter.integration.meituan.*;
import com.bxm.adscounter.model.RtbAdvertiser;
import com.bxm.adscounter.service.openlog.common.report.Reporter;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.params.ProductionCommonParam;
import com.bxm.warcar.logging.extension.CustomLoggingWriter;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.TypeHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;

/**
 * @author allen
 * @date 2022-08-22
 * @since 1.0
 */
@Configuration
public class MeituanReporter implements Reporter {

    private final static Pattern MACRO_REGX = Pattern.compile("^__.*__$");

    private final MeituanProperties meituanProperties;
    private final MeituanIntegration meituanIntegration;
    private final MeituanFeedback meituanFeedback;

    public MeituanReporter(MeituanProperties meituanProperties, MeituanIntegration meituanIntegration,
                           @Qualifier("customLoggingWriterForOcpxClickReporting") CustomLoggingWriter customLoggingWriter) {
        this.meituanProperties = meituanProperties;
        this.meituanIntegration = meituanIntegration;
        this.meituanFeedback = new DefaultMeituanFeedbackImpl(customLoggingWriter);
    }

    @Override
    public void report(KeyValueMap clickLog, String bxmFeedbackUrl) {
        String source = clickLog.getFirst("source");
        if (StringUtils.isBlank(source)) {
            return;
        }
        String idfaMd5 = clearMacroValue(clickLog.getFirst(ProductionCommonParam.IDFA_MD5));
        String imeiMd5 = clearMacroValue(clickLog.getFirst(ProductionCommonParam.IMEI_MD5));
        String oaid = clearMacroValue(clickLog.getFirst(ProductionCommonParam.OAID));
        String oaidMd5 = clearMacroValue(clickLog.getFirst(ProductionCommonParam.OAID_MD5));
        String actionTime = Optional
                .ofNullable(clickLog.getFirst("action_time"))
                .filter(StringUtils::isNotBlank)
                .orElseGet(() -> TypeHelper.castToString(System.currentTimeMillis()));

        String os = Optional.ofNullable(clickLog.getFirst(Reporter.NAME_X_OS))
                .filter(StringUtils::isNotBlank)
                .orElseGet(() -> StringUtils.isNotBlank(idfaMd5) ? "ios" : "android");

        MeituanRequest request = new MeituanRequest()
                .setSource(source)
                .setAppType(os)
                .setImeiMd5(imeiMd5)
                .setIdfaMd5(idfaMd5)
                .setOaid(oaid)
                .setOaidMd5(oaidMd5);

        request.setReportUrl(meituanProperties.getReportClickUrl());
        request.addExtend("app", "group");
        request.addExtend("coderesp", "true");
        request.addExtend("monitor_type", "click");
        request.addExtend("mt_channel", "meituanunion");
        request.addExtend("action_time", TypeHelper.castToString(actionTime));

        // 转化回调：有转化发生时，会调用渠道侧提供的回传链接(即点击上报链接中的feedback_url)，
        // 调用时会拼接上转化类型字段（event_type）,目前支持的转化类型如下，
        // 另外会拼接event_time字段，行为发生时间戳，单位毫秒
        request.setFeedbackUrl(UrlHelper.urlEncode(bxmFeedbackUrl));

        meituanIntegration.report(request, meituanFeedback);
    }

    @Override
    public RtbAdvertiser rtbAdvertiser() {
        return RtbAdvertiser.MeiTuan;
    }

    private static class DefaultMeituanFeedbackImpl implements MeituanFeedback {

        private final CustomLoggingWriter customLoggingWriter;

        private DefaultMeituanFeedbackImpl(CustomLoggingWriter customLoggingWriter) {
            this.customLoggingWriter = customLoggingWriter;
        }

        @Override
        public void call(MeituanRequest request, MeituanResponse response, String requestUri, Throwable throwable) {
            if (StringUtils.isBlank(requestUri)) {
                return;
            }
            UriComponents build = UriComponentsBuilder.fromUriString(requestUri).build();

            MultiValueMap<String, String> params = build.getQueryParams();

            Map<String, String> writeParams = Maps.newHashMap();

            for (Map.Entry<String, List<String>> entry : params.entrySet()) {
                List<String> list = entry.getValue();
                if (CollectionUtils.isNotEmpty(list)) {
                    writeParams.put(entry.getKey(), list.get(0));
                }
            }

            writeParams.put("url", requestUri);
            writeParams.put("responseBody", JsonHelper.convert(response));
            if (Objects.nonNull(throwable)) {
                writeParams.put("exception", throwable.getMessage());
            }

            customLoggingWriter.write(writeParams);
        }
    }

    private String clearMacroValue(String str) {
        if (StringUtils.isNotBlank(str) && MACRO_REGX.matcher(str).find()) {
            return StringUtils.EMPTY;
        }
        return str;
    }
}
