package com.bxm.adscounter.service.openlog.inads.listener;

import com.bxm.adscounter.model.EffectEndpoint;
import com.bxm.adscounter.service.events.EffectEvent;
import com.bxm.adscounter.service.openlog.inads.event.AdConversionEvent;
import com.bxm.adscounter.service.utils.GeneralEndpointUtils;
import com.bxm.adsprod.counter.event.TicketClickEvent;
import com.bxm.adsprod.counter.event.TicketEffectClickEvent;
import com.bxm.adsprod.facade.ticket.ClickRequest;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.integration.eventbus.AsyncEventPark;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.TypeHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.springframework.context.annotation.Configuration;

import java.util.Date;
import java.util.Objects;

/**
 * 广告转化事件监听器
 *
 * @author allen
 * @date 2022-03-22
 * @since 1.0
 */
@Slf4j
@Configuration
public class AdConversionEventListener implements EventListener<AdConversionEvent> {

    private final AsyncEventPark asyncEventPark;
    private final Fetcher fetcher;

    public AdConversionEventListener(AsyncEventPark asyncEventPark, Fetcher fetcher) {
        this.asyncEventPark = asyncEventPark;
        this.fetcher = fetcher;
    }

    @Override
    @Subscribe
    public void consume(AdConversionEvent event) {
        KeyValueMap map = event.getLog();
        postEffectEvent(map);
        postTicketClickEvent(map);
    }

    private void postEffectEvent(KeyValueMap map) {
        EffectEndpoint endpoint = GeneralEndpointUtils.fromInads(map, 0);
        asyncEventPark.post(new EffectEvent(this, endpoint));
    }

    private void postTicketClickEvent(KeyValueMap map) {
        String bxmId = map.getFirst(Inads.Param.BXMID);
        String adid = map.getFirst(Inads.Param.ADID);
        if (StringUtils.isBlank(adid)) {
            log.warn("Not found adid: {}", bxmId);
            return;
        }
        Ticket ticket = get(adid);
        if (Objects.isNull(ticket)) {
            log.warn("Not found: {}", adid);
            return;
        }
        ClickRequest req = new ClickRequest();
        req.setUid(map.getFirst(Inads.Param.UID));
        req.setTicketId(TypeHelper.castToBigInteger(adid));
        req.setPosition(map.getFirst(Inads.Param.TAGID));
        req.setValid(true);
        req.setBillid(bxmId);
        req.setAssetsId(TypeHelper.castToBigInteger(map.getFirst(Inads.Param.CREATEID)));
        req.setEntrTicketId(TypeHelper.castToBigInteger(map.getFirst(Inads.Param.EN_ADID)));
        req.setEntrAssetsId(TypeHelper.castToBigInteger(map.getFirst(Inads.Param.EN_CID)));
        req.setBidid(map.getFirst(Inads.Param.BIDID));
        req.setObidid(map.getFirst(Inads.Param.OBIDID));
        req.setAdxAppId(map.getFirst(Inads.Param.ADX_APPID));
        req.setCost(!StringUtils.equalsIgnoreCase(map.getFirst(Inads.Param.PAYED), "1"));
        req.setType(NumberUtils.toInt(map.getFirst(Inads.Param.CONVERSION_TYPE), 0));

        ticket.setSettleType(NumberUtils.toByte(map.getFirst(Inads.Param.SETTLE_TYPE), ticket.getSettleType()));
        ticket.setCpaPrice(NumberUtils.toInt(map.getFirst(Inads.Param.OCPA_OFFER_PRICE), ticket.getCpaPrice()));
        ticket.setTime(new Date());
        ticket.setOfferPrice(NumberUtils.toInt(map.getFirst(Inads.Param.OFFER_PRICE)));
        ticket.setBillid(bxmId);

        asyncEventPark.post(new TicketClickEvent(this, req, ticket));
        asyncEventPark.post(new TicketEffectClickEvent(this, req, ticket));
    }

    private Ticket get(String adid) {
        return fetcher.hfetch(TicketKeyGenerator.getAllTickets(), adid, Ticket.class);
    }
}
