package com.bxm.adscounter.service.openlog.inads.listener.ad.click;

import com.bxm.adscounter.integration.TicketServiceIntegration;
import com.bxm.adscounter.service.openlog.inads.event.AdClickEvent;
import com.bxm.adscounter.service.reporting.MacrosReportingSender;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketAppExtend;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.TypeHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Objects;

/**
 * 滴滴点击上报。
 *
 * @author allen
 * @date 2022-06-15
 * @since 1.0
 */
@Slf4j
@Configuration
public class DidiReportingAdClickEventListener implements EventListener<AdClickEvent> {

    private final TicketServiceIntegration ticketServiceIntegration;
    private final MacrosReportingSender sender;

    public DidiReportingAdClickEventListener(TicketServiceIntegration ticketServiceIntegration, MacrosReportingSender sender) {
        this.ticketServiceIntegration = ticketServiceIntegration;
        this.sender = sender;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(AdClickEvent event) {
        String basicUrl = "https://gungnir.xiaojukeji.com/data-ingestion/api/collect-common-cpa" +
                "?event_type=1" +
                "&account_id=__ACCOUNT_ID__" +
                "&product_id=__PRODUCT_ID__" +
                "&ad_target=__AD_TARGET__" +
                "&lp_type=__LP_TYPE__" +
                "&ad_team=__AD_TEAM__" +
                "&adx=__ADX__" +
                "&imei=__IMEI__" +
                "&imei_md5=__IMEI_MD5__" +
                "&idfa=__IDFA__" +
                "&idfa_md5=__IDFA_MD5__" +
                "&oaid=__OAID__" +
                "&oaid_md5=__OAID_MD5__" +
                "&device_os=__DEVICE_OS__" +
                "&request_id=__REQUEST_ID__" +
                "&level_two_media=__LEVEL_TWO_MEDIA__" +
                "&ip=__IP__" +
                "&client_time=__TIME__" +
                "&ua=__UA__" +
                "&callback_url=__CALLBACK__" +
                "&dchn=__DCHN__";
        KeyValueMap m = event.getLog();
        String adid = m.getFirst(Inads.Param.ADID);
        MultiValueMap<String, String> params = getParamForJumpUrl(adid);
        if (MapUtils.isEmpty(params)) {
            return;
        }

        String deviceOs = StringUtils.isNotBlank(m.getFirst(Inads.Param.IDFA)) || StringUtils.isNotBlank(m.getFirst(Inads.Param.IDFA_MD5))
                ? "IOS" : "android";

        basicUrl = basicUrl.replaceAll("__ACCOUNT_ID__", defaultIfBlank(params, "account_id"))
                .replaceAll("__PRODUCT_ID__", defaultIfBlank(params, "product_id"))
                .replaceAll("__AD_TARGET__", defaultIfBlank(params, "ad_target"))
                .replaceAll("__LP_TYPE__", defaultIfBlank(params, "lp_type"))
                .replaceAll("__AD_TEAM__", defaultIfBlank(params, "ad_team"))
                .replaceAll("__ADX__", defaultIfBlank(params, "adx"))
                .replaceAll("__DEVICE_OS__", deviceOs)
                .replaceAll("__LEVEL_TWO_MEDIA__", randomMedia())
                .replaceAll("__DCHN__", defaultIfBlank(params, "dchn"));

        sender.send(basicUrl, m);
    }

    private MultiValueMap<String, String> getParamForJumpUrl(String adid) {
        Ticket ticket = ticketServiceIntegration.get(TypeHelper.castToBigInteger(adid));
        if (Objects.isNull(ticket)) {
            return null;
        }
        TicketAppExtend extend = ticket.getAdTicketAppExtend();
        if (Objects.isNull(extend)) {
            return null;
        }
        String jumpUrl = extend.getJumpUrl();
        if (!StringUtils.startsWith(jumpUrl, "onetravel://router/page/web")) {
            return null;
        }
        String url = UrlHelper.getFirstValueOfParamName(jumpUrl, "url");
        if (StringUtils.isBlank(url)) {
            return null;
        }
        String decode = UrlHelper.urlDecode(url);
        return UriComponentsBuilder.fromUriString(decode).build().getQueryParams();
    }

    private String defaultIfBlank(MultiValueMap<String, String> params, String name) {
        if (params.containsKey(name)) {
            return params.getFirst(name);
        }
        return StringUtils.EMPTY;
    }

    private static final String[] MEDIA_ARRAY = new String[] {
            "com.mobile.hiweather",
            "com.qihoo360.mobilesafe",
            "com.qihoo.cleandroid_cn",
            "com.snda.wifilocating",
            "cn.wps.moffice_eng",
            "com.baidu.video",
            "com.changba",
            "com.dongqiudi.news",
            "com.douban.frodo",
            "com.handsgo.jiakao.android",
            "com.jsmcc",
            "com.kugou.android.ringtone",
            "com.hunantv.imgo.activity",
            "com.mt.mtxx.mtxx",
            "com.wondertek.hecmccmobile",
            "com.moji.mjweather",
            "fm.qingting.qtradio",
            "com.jifen.qukan",
            "com.jiayuan",
            "com.sogou.map.android.maps",
            "sogou.mobile.explorer",
            "com.sohu.sohuvideo",
            "com.ximalaya.ting.lite",
            "com.mygolbs.mybus",
            "com.zhihu.android",
            "cn.youth.news",
            "com.qiyi.video"
    };

    private static String randomMedia() {
        return MEDIA_ARRAY[RandomUtils.nextInt(MEDIA_ARRAY.length)];
    }
}
