package com.bxm.adscounter.service.openlog.inads.listener.index;

import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.rtb.common.ClickTracker;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.control.RatioControlRtbIntegration;
import com.bxm.adscounter.service.openlog.inads.event.IndexShowEvent;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import java.util.Objects;

/**
 * @author allen
 * @date 2022-11-21
 * @since 1.0
 */
@Slf4j
public abstract class AbstractSmartControlIndexShowEventListener {

    private final RtbIntegration rtbIntegration;
    private final RatioControlRtbIntegration ratioControlRtbIntegration;

    public AbstractSmartControlIndexShowEventListener(RtbIntegration rtbIntegration, RatioControlRtbIntegration ratioControlRtbIntegration) {
        this.rtbIntegration = rtbIntegration;
        this.ratioControlRtbIntegration = ratioControlRtbIntegration;
    }

    /**
     * 从中间页曝光行为日志的 ext 里获取 clickId 值的参数名
     * @return 参数名
     */
    protected abstract String getClickIdNameOnExt();

    /**
     * 从中间页曝光行为日志的 ext 里获取 ad_group_id 值的参数名。
     * 如果你希望通过其他方式获取 ad_group_id，可返回 {@code null}，然后重写 {@link #getAdGroupIdValue(KeyValueMap, JSONObject)}
     * 方法。
     * @return 参数名
     * @see #getAdGroupIdValue(KeyValueMap, JSONObject) (JSONObject)
     */
    protected abstract String getAdGroupIdNameOnExt();

    /**
     * 返回 ad_group_id 值。
     * @param keyValueMap KeyValueMap
     * @param ext ext 参数列表
     * @return ad_group_id 值
     */
    protected String getAdGroupIdValue(KeyValueMap keyValueMap, JSONObject ext) {
        return null;
    }

    /**
     * 从中间页曝光行为事件里获取 app 值
     * @param event 事件
     * @return app 标识值
     */
    protected abstract String getApp(IndexShowEvent event);

    protected void consume(IndexShowEvent event) {
        KeyValueMap keyValueMap = event.getLog();
        String ext = keyValueMap.getFirst(Inads.Param.EXT);
        if (StringUtils.isBlank(ext)) {
            return;
        }
        String clickId = null, adGroupId = null;
        try {
            JSONObject jsonObject = JSONObject.parseObject(ext);
            clickId = jsonObject.getString(getClickIdNameOnExt());
            String adGroupIdNameOnExt = getAdGroupIdNameOnExt();
            if (StringUtils.isBlank(adGroupIdNameOnExt)) {
                adGroupId = getAdGroupIdValue(keyValueMap, jsonObject);
            } else {
                adGroupId = jsonObject.getString(adGroupIdNameOnExt);
            }
        } catch (Exception e) {
            log.warn("parseObject: {}", e.getMessage());
        }
        if (StringUtils.isBlank(clickId)) {
            return;
        }

        if (StringUtils.isBlank(adGroupId) && (rtbIntegration instanceof ClickTracker)) {
            // 如果中间页没有，那么尝试从点击监测里获取。
            KeyValueMap clickTracker = ((ClickTracker) rtbIntegration).getClickTracker(clickId);
            if (Objects.nonNull(clickTracker)) {
                adGroupId = clickTracker.getFirst(Common.Param.AD_GROUP_ID);
            }
        }

        KeyValueMap clickTrackerKeyValueMap = keyValueMap.deepCopy();
        clickTrackerKeyValueMap.put(Common.Param.CLICK_ID, clickId);
        clickTrackerKeyValueMap.put(Common.Param.AD_GROUP_ID, adGroupId);
        clickTrackerKeyValueMap.put(Common.RtbParam.TAGID, keyValueMap.getFirst(Inads.Param.TAGID));

        if (rtbIntegration instanceof ClickTracker) {
            ((ClickTracker) rtbIntegration).saveClickTracker(clickTrackerKeyValueMap);
        }

        String app = getApp(event);
        ratioControlRtbIntegration.onClick(rtbIntegration, clickTrackerKeyValueMap, adGroupId, app);
    }
}
