package com.bxm.adscounter.service.openlog.pangu.listener;

import com.bxm.adscounter.integration.TicketServiceIntegration;
import com.bxm.adscounter.service.openlog.pangu.event.PanguAdWinEvent;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketAppExtend;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Pangu;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.TypeHelper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Objects;
import java.util.Optional;

/**
 * 在盘古广告胜出事件，广告成本日预算超额暂停处理器。
 *
 * @author allen
 * @date 2021-11-02
 * @since 1.0
 */
@Slf4j
@Configuration
public class PanguWinCalBudgetEventListener implements EventListener<PanguAdWinEvent> {

    private final TicketServiceIntegration ticketServiceIntegration;
    private final Counter counter;

    public PanguWinCalBudgetEventListener(TicketServiceIntegration ticketServiceIntegration, Counter counter) {
        this.ticketServiceIntegration = ticketServiceIntegration;
        this.counter = counter;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(PanguAdWinEvent event) {
        KeyValueMap keyValueMap = event.getLog();
        String adid = keyValueMap.getFirst(Pangu.Param.ADID);
        String price = keyValueMap.getFirst(Pangu.Param.OFFER_PRICE_PRECISION);
        if (Objects.isNull(adid) || Objects.isNull(price)) {
            if (log.isWarnEnabled()) {
                log.warn("Pangu compute budget fail, adid={}, price={}", adid, price);
            }
            return;
        }

        double current = Optional.ofNullable(increment(adid, price)).orElse(0D);
        BigInteger ticketId = new BigInteger(adid);
        Ticket ticket = ticketServiceIntegration.get(ticketId);
        if (Objects.isNull(ticket)) {
            log.warn("Not found ticket: {}", adid);
            return;
        }
        TicketAppExtend appExtend = ticket.getAdTicketAppExtend();
        int budget = 0;
        if (Objects.nonNull(appExtend)) {
            budget = Optional.ofNullable(appExtend.getAdxBudgetDaily()).orElse(0);
        }
        if (current >= budget && ticket.isAvailableForStatus()) {
            boolean res = ticketServiceIntegration.updateTicketStatus(ticketId, Ticket.STATUS_PAUSE, Ticket.PAUSE_REASON_OUT_OF_DAILYBUDGET);
            log.info("广告券({})状态(暂停-ADX日预算不足/日曝光)更新结果：{}", ticketId ,res);
        }
    }

    private Double increment(String adid, String price) {
        return counter.hincrFloatByAndGet(TicketKeyGenerator.Statistics.getPanGuBudgetOfDaily(), adid, new BigDecimal(price).doubleValue(), getExpireTimeInSeconds());
    }

    protected int getExpireTimeInSeconds() {
        return TypeHelper.castToInt(DateHelper.getRemainSecondsOfDay(10));
    }
}
