package com.bxm.adscounter.service.reporting;

import com.bxm.adscounter.service.autoconfigure.Properties;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.Production;
import com.bxm.openlog.sdk.consts.Inads;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Objects;
import java.util.function.Function;

/**
 * @author allen
 * @date 2022-05-30
 * @since 1.0
 */
@Slf4j
public abstract class AbstractHttpReportingSender implements ReportingSender {

    protected final Properties properties;
    protected final HttpClient httpClient;

    private ReportingLogger reportingLogger;

    public AbstractHttpReportingSender(Properties properties) {
        this.properties = properties;
        RequestConfig requestConfig = RequestConfig.custom()
                .setRedirectsEnabled(false)
                .setConnectionRequestTimeout(100)
                .setConnectTimeout(1000)
                .setSocketTimeout(2000)
                .setExpectContinueEnabled(false)
                .build();
        this.httpClient = HttpClients.custom().setDefaultRequestConfig(requestConfig).build();
    }

    @Autowired
    public void setReportingLogger(ReportingLogger reportingLogger) {
        this.reportingLogger = reportingLogger;
    }

    protected abstract String createUrl(String url, KeyValueMap keyValueMap);

    protected abstract HttpRequestBase createHttpRequestBase(String url, KeyValueMap keyValueMap);

    @Override
    public boolean send(String url, KeyValueMap keyValueMap) {
        return this.send(url, keyValueMap, s -> s);
    }

    @Override
    public boolean send(String url, KeyValueMap keyValueMap, Function<String, String> changeUrl) {
        String uri = null;
        HttpRequestBase base = null;
        try {
            uri = createRequestUri(url, keyValueMap, changeUrl);
            base = createHttpRequestBase(uri, keyValueMap);

            HttpResponse response = httpClient.execute(base);
            int statusCode = response.getStatusLine().getStatusCode();
            if (HttpStatus.SC_OK != statusCode) {
                reportingLogger.collect(uri, statusCode, null, null);
                return false;
            }
            String body = EntityUtils.toString(response.getEntity());
            reportingLogger.collect(uri, statusCode, body, null);
            return StringUtils.equals("OK", body);
        } catch (Exception e) {
            log.error("send: ", e);
            reportingLogger.collect(uri, -1, null, e);
        } finally {
            if (Objects.nonNull(base)) {
                base.releaseConnection();
            }
        }
        return false;
    }

    private String createRequestUri(String url, KeyValueMap keyValueMap, Function<String, String> changeUrl) {
        String uri = createUrl(url, keyValueMap);
        if (Objects.nonNull(changeUrl)) {
            uri = changeUrl.apply(uri);
        }
        return uri;
    }

    protected String getDefault(Object object) {
        return Objects.toString(object, "");
    }

    protected String getCallbackUrl(String bxmId) {
        KeyValueMap map = new KeyValueMap();
        map.setProduction(Production.INADS);
        map.setMt(Inads.Mt.AdConversion.original());
        map.putIfNotBlank(Inads.Param.BXMID, bxmId);
        return map.createOpenLogRequestUri(properties.getOpenLogRequestOutDomain());
    }
}
