package com.bxm.adscounter.service.scheduler;

import com.bxm.adscounter.service.scheduler.file.LogFileUploader;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileExistsException;
import org.apache.commons.io.FileUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.Scheduled;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.ScanParams;
import redis.clients.jedis.ScanResult;

import java.io.File;
import java.io.IOException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author allen
 * @date 2022-04-20
 * @since 1.0
 */
@Slf4j
@Configuration
public class VisionUploadScheduler {

    private final JedisPool jedisPool;
    private final File dir;
    private final LogFileUploader uploader;

    public VisionUploadScheduler(@Qualifier("visionJedisPool") JedisPool jedisPool, LogFileUploader uploader) {
        this.jedisPool = jedisPool;
        this.uploader = uploader;
        this.dir = this.createDir();
    }

    @Scheduled(cron = "0 0 * * * ?")
    public void execute() {
        this.scanAndUpload("inads", 0);
        this.scanAndUpload("adx", 1);
    }

    private void scanAndUpload(String primary, int db) {
        long st = System.currentTimeMillis();
        String fileName = getCurrentlyFile(primary);
        this.deleteIfExists(fileName);
        log.info("Starting...");
        try (Jedis jedis = jedisPool.getResource()) {
            jedis.select(db);
            final String start = "0";
            String cursor = start;
            ScanParams params = new ScanParams().match("CHEAT:*").count(5000);
            do {
                ScanResult<String> scan = jedis.scan(cursor, params);
                cursor = scan.getStringCursor();
                List<String> keys = scan.getResult();
                Set<String> lines = keys.stream().map(new Function<String, String>() {
                    @Override
                    public String apply(String e) {
                        String[] arr = e.split(":");
                        if (arr.length < 3) {
                            log.info("illegal key: {}", e);
                            return null;
                        }
                        return arr[2];
                    }
                }).collect(Collectors.toSet());
                writeLines(fileName, lines);
            } while (!start.equals(cursor));
            uploader.upload(new File(fileName));
            log.info("Finished in {} sec.", (System.currentTimeMillis() - st)  / 1000);
        } catch (Exception e) {
            log.error("", e);
        }
    }

    private void writeLines(String fileName, Set<String> strings) {
        try {
            File file = new File(fileName);
            if (!file.exists()) {
                if (!file.createNewFile()) {
                    throw new IOException("Cannot create file: " + file);
                }
            }
            FileUtils.writeLines(file, strings, true);
            log.info("Write size {} to {}", strings.size(), fileName);
        } catch (IOException e) {
            log.error("writeLines: ", e);
        }
    }

    private void deleteIfExists(String fileName) {
        File file = new File(fileName);
        if (file.exists()) {
            try {
                if (!file.delete()) {
                    throw new FileExistsException("unknown");
                }
            } catch (Exception e) {
                log.info("File {} can not delete! {}", fileName, e.getMessage());
            }
        }
    }

    private File createDir() {
        String fileCacheDir = System.getProperty("user.home") + File.separator + "vision-data";
        File file = new File(fileCacheDir);
        if (!file.exists()) {
            if (!file.mkdirs()) {
                throw new RuntimeException("Cannot create dir: " + file);
            }
        } else if (file.isFile()) {
            throw new RuntimeException(file + " is file!");
        }
        return file;
    }

    private String getCurrentlyFile(String primary) {
        return dir + File.separator + now() + "_" + primary + ".bxm";
    }

    private String now() {
        return LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy_MM_dd"));
    }
}
