package com.bxm.adsprod.counter.event.listeners;

import com.bxm.adsprod.counter.event.BalanceNotEnoughEvent;
import com.bxm.adsprod.facade.advertiser.AdvertiserService;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketService;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 广告主余额不足事件。对该广告主所属的所有广告券进行状态变更。
 * @author allen
 * @since V1.0.0 2017/12/15
 */
@Component
public class BalanceNotEnoughEventListener implements EventListener<BalanceNotEnoughEvent>, InitializingBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(BalanceNotEnoughEventListener.class);

    @Autowired
    private TicketService ticketService;
    @Autowired
    private AdvertiserService advertiserService;

    private final ScheduledExecutorService recover = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("recover"));
    /**
     * 暂停的广告主
     */
    private final CopyOnWriteArraySet<BigInteger> advertiserOnPause = new CopyOnWriteArraySet<>();

    @Override
    public void afterPropertiesSet() throws Exception {
        // 1分钟调度一次，查看因余额不足的广告主被暂停后，尝试查找是否加款并对广告进行恢复处理。
        recover.scheduleWithFixedDelay(() -> {
            if (CollectionUtils.isNotEmpty(advertiserOnPause)) {
                long start = System.currentTimeMillis();
                for (BigInteger id : advertiserOnPause) {
                    Long balance = advertiserService.getAdvertiserBalance(id);
                    if (null != balance && balance > 0) {
                        advertiserService.incrementBalance(id, 0L);
                        advertiserOnPause.remove(id);
                    }
                }
                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("Finished recover advertiser's ticket in {} ms", (System.currentTimeMillis() - start));
                }
            }
        }, 1, 1, TimeUnit.MINUTES);
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(BalanceNotEnoughEvent event) {
        Ticket ticket = event.getTicket();
        BigInteger advertiser = ticket.getAdvertiser();
        if (Objects.isNull(advertiser)) {
            return;
        }
        List<BigInteger> ids = ticketService.getTicketIdsByAdvertiserId(advertiser);
        if (CollectionUtils.isNotEmpty(ids)) {
            ids.forEach(id -> {
                Ticket e = ticketService.get(id);
                if (Objects.isNull(e)) {
                    return;
                }
                boolean isSameAdvertiser = null != e.getAdvertiser() && e.getAdvertiser().equals(advertiser);
                boolean isCpc = e.isAutoSettleType();
                if (isSameAdvertiser && isCpc && e.isAvailableForStatus()) {
                    Long balance = advertiserService.getAdvertiserBalance(advertiser);
                    if (null != balance && balance <= 0) {
                        Ticket t = ticketService.get(id);
                        if (t.getStatus() == Ticket.STATUS_CLOSE) {
                            return;
                        }
                        boolean flag = ticketService.updateTicketStatus(id, Ticket.STATUS_PAUSE, Ticket.PAUSE_REASON_BALANCE_NOT_ENOUGHT);

                        if (flag) {
                            if (LOGGER.isInfoEnabled()) {
                                LOGGER.info("广告券({})状态(暂停-余额不足)更新完成", id);
                            }
                        }
                        else {
                            if (LOGGER.isWarnEnabled()) {
                                LOGGER.warn("广告券({})状态(暂停-余额不足)更新失败!", id);
                            }
                        }

                        advertiserOnPause.add(advertiser);
                    }
                }
            });
        }
    }
}
