package com.bxm.adsprod.counter.event.listeners;

import com.bxm.adsprod.counter.event.BudgetNotEnoughOfHourlyEvent;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.TicketService;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.util.Calendar;

/**
 * <p>时间段预算不足事件</p>
 * @author allen
 * @since 1.0.0
 */
@Component
public class BudgetNotEnoughOfHourlyEventListener implements EventListener<BudgetNotEnoughOfHourlyEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(BudgetNotEnoughOfHourlyEventListener.class);

    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;
    @Autowired
    private TicketService ticketService;

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(BudgetNotEnoughOfHourlyEvent event) {
        Ticket ticket =  ticketService.get(event.getTicket().getId());
        if(Ticket.STATUS_CLOSE==ticket.getStatus()){
            return;
        }

        if (!ticket.isAvailableForStatus()) {
            return;
        }

        BigInteger id = ticket.getId();

        boolean semaphore = event.isSemaphore();
        if (semaphore) {
            this.setSemaphore(id, event.getTrigger(), event.getSegmentStart(), event.getSegmentEnd());
        }

        boolean flag = ticketService.updateTicketStatus(id, Ticket.STATUS_PAUSE, Ticket.PAUSE_REASON_OUT_OF_TIMEBUDGET);

        if (flag) {
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("广告券({})状态(暂停-时间段预算不足)更新完成", id);
            }
        }
        else {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("广告券({})状态(暂停-时间段预算不足)更新失败!", id);
            }
        }
    }

    /**
     * 设置一个信号，让任务知道在执行时间是否需要重新打开
     * @param id
     * @param trigger
     * @param segmentStart
     * @param segmentEnd
     */
    private void setSemaphore(BigInteger id, int trigger, int segmentStart, int segmentEnd) {
        // TODO xx:xx:59 过期，让状态刷新任务在 xx:xx:00 不会读到脏数据。但是这种做法是不严谨的，后续需要优化
        long expire = getRemainSecondsOfHour(segmentEnd - trigger) - 1;
        if (expire <= 0) {
            return;
        }
        String value = StringUtils.join(new Object[] { trigger, segmentStart, segmentEnd }, '-');
        updater.update(TicketKeyGenerator.Temp.getTimelineOutOfBudget(id), value, (int) expire);
    }

    private static long getRemainSecondsOfHour(int hour) {
        return (60 * 60 * hour) - DateUtils.getFragmentInSeconds(Calendar.getInstance(), Calendar.HOUR_OF_DAY);
    }
}
