package com.bxm.adsprod.counter.event.listeners;

import com.bxm.adsprod.counter.event.TicketClickEvent;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.localdate.LocalDateTimeHelper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Configuration;

import java.time.LocalDateTime;


/**
 * <h3>时间滑动窗口点击均价</h3>
 *
 * @author hcmony
 * @since V1.0.0, 2020/8/10 10:39
 */
@Configuration
public class ClickAveragePriceEventListener implements EventListener<TicketClickEvent> {

    @Qualifier("jedisUpdater")
    @Autowired
    private Updater updater;

    @Qualifier("jedisCounter")
    @Autowired
    private Counter counter;

    @Subscribe
    @Override
    @AllowConcurrentEvents
    public void consume(TicketClickEvent event) {

        Ticket ticket = event.getTicket();
        if (!ticket.isAutoSettleType()) {
            return ;
        }

        if (event.getRequest().isValid()) {
            return ;
        }

        // 1，点击
        increment(TicketKeyGenerator.TIMENODE.currentTimeNodeClick(currentTimeNode()), String.valueOf(ticket.getId()), 1L);

        // 2，消耗
        final long offerPrice = getOfferPrice( ticket);
        increment(TicketKeyGenerator.TIMENODE.currentTimeNodeConsume(currentTimeNode()), String.valueOf(ticket.getId()), offerPrice);
    }


    private void increment(KeyGenerator key, String field, long inc) {
        counter.hincrementByAndGet(key, field, inc, getExpireTimeInSeconds());
    }

    private int getExpireTimeInSeconds() {
        return 60 * 60 * 2;
    }

    private long getOfferPrice( Ticket ticket) {
        Integer offerPrice = ticket.getOfferPrice();
        return (ticket.isAutoSettleType() ? (null == offerPrice) ? ticket.getPrice() : offerPrice : 0);
    }

    private String timeNode(LocalDateTime now) {
        return LocalDateTimeHelper.formatToString(now, LocalDateTimeHelper.PATTERN_STR8) + (now.getHour() * 60 + now.getMinute()) / 10;
    }

    private String currentTimeNode() {
        return timeNode(LocalDateTime.now());
    }

}
