package com.bxm.adsprod.counter.event.listeners;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bxm.adscounter.integration.adsmanager.AdsmanagerService;
import com.bxm.adsprod.counter.event.TicketClickEvent;
import com.bxm.adsprod.facade.advertiser.AdvertiserConstant;
import com.bxm.adsprod.facade.advertiser.StationUserWeightPushParam;
import com.bxm.adsprod.facade.ticket.ClickRequest;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.MessageSender;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.curator.shaded.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * Description CPC券消耗时， 站内户权重的一些操作
 *
 * @author hxpeng
 * <p>
 * Date 2018/12/19 9:54
 */
@Component
public class StationUserWeightTicketConsumeListener implements EventListener<TicketClickEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(StationUserWeightTicketConsumeListener.class);

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;
    @Autowired
    @Qualifier("jedisCounter")
    protected Counter counter;
    @Autowired
    private AdsmanagerService adsmanagerService;
    @Autowired
    private MessageSender messageSender;

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(TicketClickEvent ticketClickEvent) {
        ClickRequest clickRequest = ticketClickEvent.getRequest();
        Ticket ticket = ticketClickEvent.getTicket();
        // 非cpc券的点击 不要
        if (!ticket.isAutoSettleType()) {
            return;
        }
        // 根据券广告主ID 获取 站内户权重计划缓存
        String advertiserId = ticket.getAdvertiser().toString();
        String stationUserWeightListJsonStr = fetcher.hfetch(TicketKeyGenerator.Advertiser.getStationUserWeight(), advertiserId, String.class);
        if (StringUtils.isBlank(stationUserWeightListJsonStr)) {
            return;
        }
        List<StationUserWeightPushParam> stationUserWeightPushParams = JSONArray.parseArray(stationUserWeightListJsonStr, StationUserWeightPushParam.class);
        if (CollectionUtils.isEmpty(stationUserWeightPushParams)) {
            return;
        }
        Integer offerPrice = ticket.getOfferPrice();
        Integer incrementValue = (ticket.isAutoSettleType() ? (null == offerPrice) ? ticket.getPrice() : offerPrice : 0);

        String ticketIdStrRegex = AdvertiserConstant.BaseCharacter.COMMA + ticket.getId().toString() + AdvertiserConstant.BaseCharacter.COMMA;
        String positionIdStrRegex = AdvertiserConstant.BaseCharacter.COMMA + clickRequest.getPosition() + AdvertiserConstant.BaseCharacter.COMMA;

        boolean needUpdateFlag = false;
        List<StationUserWeightPushParam> updateEntities = Lists.newArrayList();
        Long completeWeightId = null;
        for (StationUserWeightPushParam weightPushParam : stationUserWeightPushParams) {
            updateEntities.add(weightPushParam);
            // 权重计划是否进行中
            if (!AdvertiserConstant.StationUserWeight.STATION_USER_WEIGHT_STATUS_NORMAL.equals(weightPushParam.getStatus())) {
                continue;
            }
            // 当前消耗券ID 是否 在监控范围内
            if (!weightPushParam.getTicketIdsStr().contains(ticketIdStrRegex)) {
                continue;
            }
            // 当前消耗券ID的广告位ID 是否 在监控范围内
            if (weightPushParam.getLimitPositionFlag()) {
                if (!weightPushParam.getPositionIdsStr().contains(positionIdStrRegex)) {
                    continue;
                }
            }
            // 记录消耗
            Long consume = counter.hincrementByAndGet(TicketKeyGenerator.Advertiser.getConsumeForStationUserWeight(), weightPushParam.getStationUserWeightId().toString(), incrementValue);
            if (consume.compareTo(weightPushParam.getWeightPlanConsume().longValue()) >= 0) {
                updateEntities.remove(updateEntities.size() - 1);
                completeWeightId = weightPushParam.getStationUserWeightId();
                // 需要更新缓存
                needUpdateFlag = true;
            }
        }
        // 更新缓存
        if (needUpdateFlag) {
            updater.hupdate(TicketKeyGenerator.Advertiser.getStationUserWeight(), advertiserId, JSON.toJSONString(updateEntities));
        }
        if (null != completeWeightId) {
            this.noticeAdsmanagerComplete(completeWeightId);
        }
    }

    /**
     * Description: 通知运营后台完成
     * JDK version used:<JDK1.8>
     * Create Date：2018/12/24 14:15
     *
     * @param stationUserWeightId 权重计划完成
     * @return void
     * @author hxpeng
     */
    private void noticeAdsmanagerComplete(Long stationUserWeightId) {
        boolean successFlag = false;
        try {
            LOGGER.info("推送 站内户权重计划完成得ID到运营后台：{}", stationUserWeightId);
            successFlag = adsmanagerService.stationUserWeightComplete(stationUserWeightId);
        } catch (Exception e) {
            LOGGER.error("发送失败！", e);
        }
        if (!successFlag) {
            messageSender.send2(new Message("权重计划已完成， 但是推送到运营后台失败了！权重id：" + stationUserWeightId));
        }
    }

}
