package com.bxm.adsprod.counter.event.listeners;

import com.bxm.adsprod.counter.event.TicketClickEvent;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Configuration;

import javax.annotation.Resource;
import java.math.BigDecimal;


/**
 * 阈值 限价处理器
 * @author allen
 * @date 2019/6/4
 * @since 1.0.0
 */
@Configuration
public class ThresholdLimitPriceEventListener implements EventListener<TicketClickEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ThresholdLimitPriceEventListener.class);
    @Resource(name = "jedisFetcher")
    private Fetcher fetcher;

    @Resource(name = "jedisUpdater")
    private Updater updater;

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(TicketClickEvent event) {
        Ticket ticket = event.getTicket();
        if (!ticket.isAutoSettleType()) {
            return;
        }
        this.updateLimitPriceEnabled(ticket);
    }

    private void updateLimitPriceEnabled(Ticket ticket) {
        String ticketId = String.valueOf(ticket.getId());
        String config = fetcher.hfetch(TicketKeyGenerator.Threshold.getLimitPriceConfig(),
                ticketId,
                String.class);
        if (StringUtils.isBlank(config)) {
            return;
        }

        String[] strings = StringUtils.split(config, "-");
        if (strings.length < 2) {
            return;
        }
        long online = NumberUtils.toLong(strings[0]);
        long offline = NumberUtils.toLong(strings[1]);
        if (online == 0 || offline == 0) {
            return;
        }
        Long clicks = fetcher.hfetch(TicketKeyGenerator.Statistics.getClickOfDaily(), ticketId, Long.class);
        if (null == clicks || 0 == clicks) {
            return;
        }
        Long budgetOfToday = fetcher.hfetch(TicketKeyGenerator.Statistics.getBudgetOfDaily(), ticketId, Long.class);
        if (null == budgetOfToday || 0 == budgetOfToday) {
            return;
        }
        long avgClickPrice = new BigDecimal(budgetOfToday).divide(new BigDecimal(clicks), 3, BigDecimal.ROUND_HALF_UP).longValue();

        String action = "NONE";
        if (avgClickPrice < offline) {
            // 低于下限阈值，关闭限价
            this.removeInEnabled(ticketId);
            action = "OFF";
        } else {
            if (avgClickPrice > online) {
                // 高于上限阈值，开启或更新限价
                this.updateInEnabled(ticketId, clicks, avgClickPrice);
                action = "ON";
            } else if (avgClickPrice <= online && isExistsInEnabled(ticketId)) {
                // 出于上下限区间，并且当前为开启状态，才更新限价
                this.updateInEnabled(ticketId, clicks, avgClickPrice);
                action = "MODIFY";
            }
        }

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("广告券({})限价{}处理完成。{}={}/{} - on:{}, off:{}", ticketId, action,
                    avgClickPrice, budgetOfToday, clicks, online, offline);
        }
    }

    private void removeInEnabled(String ticketId) {
        updater.hremove(TicketKeyGenerator.Threshold.getLimitPriceEnabled(), ticketId);
    }

    private void updateInEnabled(String ticketId, Long clicks, long avgClickPrice) {
        Long views = fetcher.hfetch(TicketKeyGenerator.Statistics.getViewOfDaily(), ticketId, Long.class);
        if (null == views || 0 == views) {
            return;
        }
        double clickRatio = new BigDecimal(clicks).divide(new BigDecimal(views), 3, BigDecimal.ROUND_HALF_UP).doubleValue();
        updater.hupdate(TicketKeyGenerator.Threshold.getLimitPriceEnabled(), ticketId, avgClickPrice + "-" + clickRatio, 24 * 60 * 60);
    }

    private boolean isExistsInEnabled(String ticketId) {
        String value = fetcher.hfetch(TicketKeyGenerator.Threshold.getLimitPriceEnabled(), ticketId, String.class);
        return StringUtils.isNotBlank(value);
    }
}
