package com.bxm.adsprod.counter.ticket;

import com.bxm.adsprod.facade.ticket.CounterRequest;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.TypeHelper;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.math.BigInteger;

/**
 * <h3>抽象的统计器</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/11
 */
public abstract class AbstractCounter<REQUEST, RETURNING> {

	 public static final Logger LOGGER = LoggerFactory.getLogger(AbstractCounter.class);

    protected static final long DEFAULT_INCREMENT_VALUE = 1;

    @Autowired
    @Qualifier("jedisCounter")
    protected Counter counter;

    /**
     * 转换Request对象
     * @param request request
     * @return REQUEST
     */
    protected abstract REQUEST convertRequest(Object request);

    /**
     * 会首先从{@link #getKeyGenerator(Object, Object)}获取，如果返回为<code>null</code> 则使用这个返回存储需要的KeyGenerator实现
     * @param uid uid
     * @param ticketId ticketId
     * @return k
     * @see #getKeyGenerator(Object, Object)
     */
    protected abstract KeyGenerator getKeyGenerator(String uid, BigInteger ticketId);

    /**
     * 返回存储需要的KeyGenerator实现，可以覆盖这个方法。
     * @param request request object
     * @param returning returning object
     * @return k
     * @see #getKeyGenerator(String, BigInteger)
     */
    protected KeyGenerator getKeyGenerator(REQUEST request, RETURNING returning) {
        return null;
    }

    /**
     * 返回增量值
     * @param request request object
     * @param returning returning object
     * @return
     */
    protected abstract long getIncrementValue(REQUEST request, RETURNING returning);

    /**
     * 返回UID
     * @param request
     * @param returning
     * @return
     */
    protected abstract String getUid(REQUEST request, RETURNING returning);

    /**
     * 返回广告券ID
     * @param request
     * @param returning
     * @return
     */
    protected abstract BigInteger getTicketId(REQUEST request, RETURNING returning);

    /**
     * 返回过期时间，默认为10天
     * @return
     */
    protected int getExpireTimeInSeconds() {
        return TypeHelper.castToInt(DateHelper.getRemainSecondsOfDay(10));
    }

    /**
     * 处理完成之前需要执行的自定义方法
     * @param val 增量的值
     * @param request
     * @param returning
     */
    protected void beforeIncrement(long val, REQUEST request, RETURNING returning) {}

    /**
     * 处理完成之后需要执行的自定义方法
     * @param val 处理完成之后的值
     * @param request
     * @param returning
     */
    protected void afterIncrement(long val, REQUEST request, RETURNING returning) {}

    /**
     * 是否需要执行increment
     * @description
     * @author 阿导
     * @time 2018/3/29
     * @CopyRight 杭州微财科技有限公司
     * @param request
     * @param returning
     * @return boolean
     */
    protected boolean isIncrementIfNeeded(REQUEST request, RETURNING returning) {
        return true;
    }

    protected final void consume(REQUEST request, RETURNING returning) {
        long incrementValue = getIncrementValue(request, returning);
        String uid = getUid(request, returning);
        BigInteger ticketId = getTicketId(request, returning);
        beforeIncrement(incrementValue, request, returning);

        Long rst = 0L;
        if (isIncrementIfNeeded(request, returning)) {
            KeyGenerator keyGenerator = getKeyGenerator(request, returning);
            if (null == keyGenerator) {
                keyGenerator = getKeyGenerator(uid, ticketId);
            }
            rst = increment(keyGenerator, request, incrementValue);
            // 其他增量
            incrementOther(request, returning, incrementValue);
        }
        afterIncrement(rst, request, returning);
    }

    /**
     * 其他增量
     */
    protected void incrementOther(REQUEST request, RETURNING returning, long incrementValue){
    }

    private Long increment(KeyGenerator keyGenerator, REQUEST r, long inc) {
        if (this instanceof HashCounter && r instanceof CounterRequest) {
            String field = ((HashCounter) this).getField((CounterRequest) r);
            if (StringUtils.isBlank(field)) {
                if (LOGGER.isWarnEnabled()) {
                    LOGGER.warn("field cannot be null! {}", r.toString());
                }
                return 0L;
            }
            return counter.hincrementByAndGet(keyGenerator, field, inc, getExpireTimeInSeconds());
        }
        else {
            return counter.incrementByAndGet(keyGenerator, inc, getExpireTimeInSeconds());
        }
    }
}
