package com.bxm.adsprod.counter.ticket.cashier;

import com.bxm.adsprod.facade.ticket.ClickRequest;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.math.BigInteger;

/**
 * <h3>广告主收银员，每一次点击扣款一次</h3>
 * <p>该实现需要结合{@link com.bxm.adsprod.facade.advertiser.AdvertiserService}使用</p>
 *
 * @see com.bxm.adsprod.facade.advertiser.AdvertiserService#incrementBalance(BigInteger, Long)
 * @author allen
 * @since V1.0.0 2017/12/15
 */
public abstract class AdvertiserCashier {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdvertiserCashier.class);
    @Autowired
    @Qualifier("jedisCounter")
    private Counter counter;
    @Autowired
    private EventPark eventPark;

    protected abstract void afterConsume(ClickRequest request, Ticket ticket, long inc, long afterInc);

    private long getIncrementValue(ClickRequest request, Ticket ticket) {
        //有效点击但不是ocpa的券   或者 ocpa的券，已扣费（点击的时候都当做已扣费），都不算价格
        if ((request.isValid() && !ticket.isOCPA())
                || (ticket.isOCPA() && request.isCost())) {
            return 0;
        }
        //当是ocpa 并且是有效点击，并且类型对的上的时候,也没产生扣费过 开始计算价格
        if (request.isValid() && ticket.isOCPA() && !request.isCost()) {
            int offerPrice = ticket.getCpaPrice();
            return - offerPrice;
        }
        Integer offerPrice = ticket.getOfferPrice();
        return - (ticket.isAutoSettleType() ? (null == offerPrice) ? ticket.getPrice() : offerPrice : 0);
    }

    protected void consume(ClickRequest request, Ticket ticket) {
        long incrementValue = getIncrementValue(request, ticket);
        long remain = increment(request.getUid(), ticket, incrementValue);
        try {
            afterConsume(request, ticket, incrementValue, remain);
        } catch (Exception e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("afterConsume: ", e);
            }
        }
    }

    private long increment(String uid, Ticket ticket, long inc) {
        KeyGenerator keyGenerator = TicketKeyGenerator.Advertiser.getBalance();
        return counter.hincrementByAndGet(keyGenerator, String.valueOf(ticket.getAdvertiser()), inc);
    }
}
