package com.bxm.adsprod.counter.ticket.counter;

import com.bxm.adsprod.counter.event.TicketClickEvent;
import com.bxm.adsprod.counter.ticket.AbstractClickCounter;
import com.bxm.adsprod.counter.ticket.HashCounter;
import com.bxm.adsprod.facade.tag.Tag;
import com.bxm.adsprod.facade.ticket.ClickRequest;
import com.bxm.adsprod.facade.ticket.CounterRequest;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.integration.pair.Pair;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigInteger;
import java.util.List;
import java.util.Set;

/**
 * <h3>日点击次数统计</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/08
 */
@Component
public class TicketDailyClickCounter extends AbstractClickCounter implements HashCounter, EventListener<TicketClickEvent> {

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;
    @Autowired
    private Pair pair;

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(TicketClickEvent event) {
        super.consume(event.getRequest(), event.getTicket());
    }

    @Override
    protected long getIncrementValue(ClickRequest clickRequest, Ticket ticket) {
        return clickRequest.isValid() ? 0 : DEFAULT_INCREMENT_VALUE;
    }

    @Override
    protected KeyGenerator getKeyGenerator(String uid, BigInteger ticketId) {
        return TicketKeyGenerator.Statistics.getClickOfDaily();
    }

	@Override
	public <T extends CounterRequest> String getField(T request) {
		ClickRequest click = convertRequest(request);
		return String.valueOf(click.getTicketId());
	}
	
	@Override
    protected void afterIncrement(long val, ClickRequest request, Ticket ticket) {
	    BigInteger ticketId = ticket.getId();
	    if(request.isValid()){
	        Object clientOriginal = super.counter.getClientOriginal();
	        if(clientOriginal instanceof JedisPool){
	            JedisPool jedisPool = (JedisPool) clientOriginal;
	            KeyGenerator key = TicketKeyGenerator.Statistics.getValidClickOfDaily(ticketId);
	            Jedis jedis = null;
	            try {
	                jedis = jedisPool.getResource();
	                jedis.pfadd(key.generateKey(),request.getBillid());
	                //过期时间设置为两天
	                jedis.expire(key.generateKey(), getExpireTimeOfDay(2));

                    if(null != request.getEntrTicketId()){
                        String keyPanGu = TicketKeyGenerator.Statistics.getPanGuValidClickOfDaily(request.getEntrTicketId()).generateKey();
                        jedis.pfadd(keyPanGu,request.getBillid());
                        //过期时间设置为两天
                        jedis.expire(keyPanGu, getExpireTimeOfDay(2));
                    }
                } catch (Exception e) {
                    LOGGER.error("实时统计券有效点击报错" + e.getMessage(),e);
                }finally{
                    if (null != jedis) {
                        jedis.close();
                    }
                }
	        }
	    }

        this.saveValidClickIfInSpecialTags(request, ticket);
    }

    /**
     * 保存这个有效点击，如果它符合特殊标签条件的话。
     *
     * @param request 点击请求
     * @param ticket 点击的广告
     */
    private void saveValidClickIfInSpecialTags(ClickRequest request, Ticket ticket) {
        if (!request.isValid()) {
            return;
        }
        String uid = request.getUid();
        if (StringUtils.isBlank(uid)) {
            return;
        }
        List<Tag> tags = ticket.getTags();
        if (CollectionUtils.isEmpty(tags)) {
            LOGGER.warn("{} ticket's tag is null!", ticket.getId());
            return;
        }

        List<String> specialTags = getTags();
        if (CollectionUtils.isEmpty(specialTags)) {
            return;
        }

        int expire = getExpire();
        for (Tag tag : tags) {
            String tagCode = tag.getCode();
            if (!specialTags.contains(tagCode)) {
                continue;
            }
            // TODO 先查后更新会存在并发问题，考虑到单个用户存在的几率低，先不考虑优化了
            Set existed = fetcher.hfetchWithSelector(TicketKeyGenerator.User.getValidClickForTag(uid), tagCode, Set.class, 4);
            if (null == existed) {
                existed = Sets.newHashSet();
            }
            existed.add(String.valueOf(ticket.getId()));
            updater.updateWithSelector(TicketKeyGenerator.User.getValidClickForTagOfLast(uid), pkgLastValue(ticket, tag), expire, 4);
            updater.hupdateWithSelector(TicketKeyGenerator.User.getValidClickForTag(uid), tagCode, existed, expire, 4);

            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Uid={},Tag={},Ticket={} has been saved!", uid, tagCode, ticket.getId());
            }
            break;
        }
    }

    private String pkgLastValue(Ticket ticket, Tag tag) {
        return StringUtils.join(new Object[] { ticket.getId(), tag.getCode() }, "-");
    }

    private List<String> getTags() {
        return pair.get("SPECIAL_TAGS").ofArrayList();
    }

    private int getExpire() {
        return 30 * 24 * 60 * 60;
    }
}
