package com.bxm.adsprod.counter.ticket.counter;

import com.bxm.adsprod.counter.event.BudgetNotEnoughOfDailyEvent;
import com.bxm.adsprod.counter.event.TicketViewEvent;
import com.bxm.adsprod.counter.ticket.AbstractViewCounter;
import com.bxm.adsprod.counter.ticket.HashCounter;
import com.bxm.adsprod.facade.media.PositionGroupService;
import com.bxm.adsprod.facade.strategy.UniformSpeedStrategyFacadeService;
import com.bxm.adsprod.facade.ticket.*;
import com.bxm.adsprod.model.so.rules.PositionGroupRuleSo;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.util.Objects;

/**
 * <h3>CPA广告券日预算统计器</h3>
 * <p>
 *     CPA 每一次预算触发是：曝光，因此它消费的是广告券曝光主题，并且在内部需要对CPC 的广告券计数增量修改为：0
 * </p>
 * <p>当日预算不足时，会对该广告券的状态更新为暂停</p>
 *
 * @author allen
 * @since V1.0.0 2017/12/11
 */
@Component
public class TicketForCpaDailyBudgetCounter extends AbstractViewCounter implements HashCounter, EventListener<TicketViewEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketForCpaDailyBudgetCounter.class);
    @Autowired
    private EventPark eventPark;
    @Autowired
    private PositionGroupService positionGroupService;
    @Autowired
    private UniformSpeedStrategyFacadeService uniformSpeedStrategyFacadeService;
    @Autowired(required = false)
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(TicketViewEvent event) {
        super.consume(event.getRequest(), event.getTicket());
    }

    @Override
	public <T extends CounterRequest> String getField(T request) {
		ViewRequest view = convertRequest(request);
		return String.valueOf(view.getTicketId());
	}

    @Override
    protected long getIncrementValue(ViewRequest ticketRequest, Ticket ticket) {
        return ticket.isAutoSettleType() ? 0 : DEFAULT_INCREMENT_VALUE;
    }

    @Override
    protected KeyGenerator getKeyGenerator(String uid, BigInteger ticketId) {
        return TicketKeyGenerator.Statistics.getBudgetOfDaily();
    }

    /**
     * @param val          增量的值
     * @param clickRequest
     * @param ticket
     * @see TicketForCpcDailyBudgetCounter#beforeIncrement(long, ClickRequest, Ticket)
     */
    @Override
    protected void beforeIncrement(long val, ViewRequest clickRequest, Ticket ticket) {
        String position = clickRequest.getPosition();
        BigInteger ticketId = ticket.getId();
        positionGroupService.incrementIfNecessary(ticketId, position, val);
        this.uniformSpeedStrategyFacadeService.consume(Objects.toString(ticket.getId()), val);
    }

    @Override
    protected void afterIncrement(long val, ViewRequest viewRequest, Ticket ticket) {
        // val = budgetOfToday
        if (val >= ticket.getBudgetDaily().longValue()) {
            // Post message to event park.
            eventPark.post(new BudgetNotEnoughOfDailyEvent(this, ticket));
        }
        this.handleClosed(viewRequest, ticket);
    }

    private void handleClosed(ViewRequest viewRequest, Ticket ticket) {
        try {
            String positionId = viewRequest.getPosition();
            BigInteger ticketId = ticket.getId();
            PositionGroupRuleSo.Entry config = fetcher.hfetch(TicketKeyGenerator.Filter.getPositionGroup(ticketId), positionId, PositionGroupRuleSo.Entry.class);
            if (null == config) {
                return;
            }
            String groupId = config.getGroupId();
            long limit = config.getLimit();

            Long budget = positionGroupService.getBudgetOfToday(ticketId, groupId);
            if (budget >= limit && limit > 0) {
                // closed
                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("Closed [{}] at {}, cfg: groupId={},limit={},budget={}",
                            ticketId, groupId, groupId, limit, budget);
                }
                updater.supdate(TicketKeyGenerator.Position.getClosedTicket(groupId), 24 * 60 * 60, String.valueOf(ticketId));
            }
        } catch (Exception e) {
            LOGGER.error("handleClosed: ", e);
        }
    }
}
