package com.bxm.adsprod.counter.ticket.counter;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.adsprod.counter.event.BudgetNotEnoughOfDailyEvent;
import com.bxm.adsprod.counter.event.TicketClickEvent;
import com.bxm.adsprod.counter.event.TicketCostExceedCostWarnEvent;
import com.bxm.adsprod.counter.ticket.AbstractClickCounter;
import com.bxm.adsprod.counter.ticket.HashCounter;
import com.bxm.adsprod.facade.media.PositionGroupService;
import com.bxm.adsprod.facade.strategy.UniformSpeedStrategyFacadeService;
import com.bxm.adsprod.facade.ticket.*;
import com.bxm.adsprod.model.so.rules.PositionGroupRuleSo;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.localdate.LocalDateTimeHelper;
import com.fasterxml.jackson.databind.JsonNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * <h3>CPC广告券日预算统计器</h3>
 * <p>
 *     CPC 每一次预算触发是：点击，因此它消费的是广告券点击主题，并且在内部需要对CPA 的广告券计数增量修改为：0
 * </p>
 * <p>当日预算不足时，会对该广告券的状态更新为暂停</p>
 *
 * @author allen
 * @since V1.0.0 2017/12/11
 */
@Component
@Slf4j
public class TicketForCpcDailyBudgetCounter extends AbstractClickCounter implements HashCounter, EventListener<TicketClickEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketForCpcDailyBudgetCounter.class);

    @Autowired
    private EventPark eventPark;
    @Autowired
    private PositionGroupService positionGroupService;
    @Autowired
    private UniformSpeedStrategyFacadeService uniformSpeedStrategyFacadeService;

    @Autowired(required = false)
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Autowired
    private TicketPositionTimesLimitService ticketPositionTimesLimitService;

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(TicketClickEvent event) {
        super.consume(event.getRequest(), event.getTicket());
    }

    @Override
	public <T extends CounterRequest> String getField(T request) {
		ClickRequest click = convertRequest(request);
		return String.valueOf(click.getTicketId());
	}

    @Override
    protected long getIncrementValue(ClickRequest clickRequest, Ticket ticket) {
        //有效点击但不是ocpa的券   或者 ocpa的券，已扣费（点击的时候都当做已扣费），都不算价格
        if ((clickRequest.isValid() && !ticket.isOCPA())
                ||  (ticket.isOCPA() && clickRequest.isCost())) {
            return 0;
        }
        //当是ocpa 并且是有效点击，并且类型对的上的时候,也没产生扣费过 开始计算价格
        if(clickRequest.isValid() && ticket.isOCPA() && !clickRequest.isCost()){
            return ticket.getCpaPrice();
        }

        Integer offerPrice = ticket.getOfferPrice();
        return (ticket.isAutoSettleType() ? (null == offerPrice) ? ticket.getPrice() : offerPrice : 0);
    }

    /**
     * @param val          增量的值
     * @param clickRequest
     * @param ticket
     * @see TicketForCpaDailyBudgetCounter#beforeIncrement(long, ViewRequest, Ticket)
     */
    @Override
    protected void beforeIncrement(long val, ClickRequest clickRequest, Ticket ticket) {
        String position = clickRequest.getPosition();
        BigInteger ticketId = ticket.getId();
        positionGroupService.incrementIfNecessary(ticketId, position, val);
        processPositionTicketTimesLimit(val, clickRequest, ticket);
        this.uniformSpeedStrategyFacadeService.consume(Objects.toString(ticket.getId()), val);
    }

    @Override
    protected void incrementOther(ClickRequest clickRequest, Ticket ticket, long incrementValue) {
        if(null != clickRequest.getEntrTicketId()){
            String field = String.valueOf(clickRequest.getEntrTicketId());
            counter.hincrementByAndGet(TicketKeyGenerator.Statistics.getPanGuCostOfDaily(),
                    field, incrementValue, getExpireTimeInSeconds());
        }

        // 素材消耗
        if (null != clickRequest.getAssetsId()) {
            String field = String.valueOf(clickRequest.getAssetsId());
            counter.hincrementByAndGet(TicketKeyGenerator.Statistics.getAssetsBudgetOfDaily(DateHelper.format("yyyyMMdd")),
                    field, incrementValue, getExpireTimeInSeconds());
        }
    }

    @Override
    protected void afterIncrement(long val, ClickRequest clickRequest, Ticket ticket) {
        // val = budgetOfToday
        if (val >= ticket.getBudgetDaily().longValue()) {
            // Post message to event park.
            eventPark.post(new BudgetNotEnoughOfDailyEvent(this, ticket));
        }
        this.handleClosed(clickRequest, ticket);
        this.handleNewCostExceedWarning( val, clickRequest ,ticket);
        this.handleExploreConsume(clickRequest ,ticket);

    }
    /**
     * 消耗提醒的redis  key 为券id  value为多个消耗提醒的列表
     * @return
     */
    public KeyGenerator getTicketCost(){
        return () -> KeyBuilder.build("AD", "TICKET", "NEWCOSWARN");
    }

    /**
     * 当天券具体的消耗 key  为 消耗提醒的id   value  为 具体消耗
     * @return
     */
    public KeyGenerator getTicketCostConsume(Long ticketId,String date){
        return () -> KeyBuilder.build("AD", "TICKET", "NEWCOST","CONSUME",date,ticketId);
    }


    public KeyGenerator getTicketCostCosredis(Long ticketId,String date){
        return () -> KeyBuilder.build("AD", "TICKET", "NEWCOST","COSREDIS",date,ticketId);
    }

    /**
     * 加速探索消耗与点击统计
     * @param clickRequest
     * @param ticket
     */
    private void handleExploreConsume(ClickRequest clickRequest, Ticket ticket) {
        try {
            //有效点击不处理
            if(clickRequest.isValid()){
                return;
            }
            //不是ocpc的券 不处理
            if(!ticket.isOcpc()){
                return;
            }
            //不是加速探索消耗
            if(!ticket.isExplorePrice()){
                return;
            }
            String ticketId = ticket.getId().toString();
            String positionId=clickRequest.getPosition();
            //开始统计埋点   全局维度是field是券ID，广告位维度field是广告位
            // （消耗相关）
            counter.hincrementByAndGet(TicketKeyGenerator.ticketExploreBudget(ticketId), ticketId, ticket.getOfferPrice(), getExpireTimeInSeconds());
            counter.hincrementByAndGet(TicketKeyGenerator.ticketExploreBudget(ticketId), positionId, ticket.getOfferPrice(), getExpireTimeInSeconds());

            //点击相关
            counter.hincrementByAndGet(TicketKeyGenerator.ticketExploreClick(ticketId), ticketId, 1, getExpireTimeInSeconds());
            counter.hincrementByAndGet(TicketKeyGenerator.ticketExploreClick(ticketId), positionId, 1, getExpireTimeInSeconds());



        } catch (Exception e) {
            LOGGER.error("券消耗提醒记录添加失败", e);
        }
    }

    private void handleNewCostExceedWarning(long val, ClickRequest clickRequest, Ticket ticket) {
        try {
            String result = fetcher.hfetch(getTicketCost(), ticket.getId() + "", String.class);
            if (StringUtils.isBlank(result)) {
                return;
            }
            String date = LocalDateTimeHelper.getDateStringNow();
            List<AdTicketCost> list = JSON.parseArray(result, AdTicketCost.class);
            if (CollectionUtils.isEmpty(list)) {
                return;
            }
            LocalDateTime midnight = LocalDateTime.now().plusDays(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
            long seconds = ChronoUnit.SECONDS.between(LocalDateTime.now(), midnight);
            //开始循环消耗提醒，检查是否钉钉提醒
            for(AdTicketCost adTicketCost : list){
                //提醒阀值
                Double cosWarn = Double.valueOf(adTicketCost.getCosWarn());
                //此处缓存用来记录这次是否需要插入券消耗提醒记录的 -> 提醒阈值+上次券消耗
                Double costRdis = fetcher.hfetch(getTicketCostConsume(adTicketCost.getTicketId(),date),  adTicketCost.getId().toString(), Double.class);
                if(costRdis == null){
                    //代表是第一次，需要初始化(意思是到这个阀值就提醒)
                    updater.hupdate(getTicketCostConsume(adTicketCost.getTicketId(),date),
                            adTicketCost.getId().toString(), cosWarn, (int) seconds);
                    continue;
                }
                if(val < costRdis){
                    continue;
                }
                Double finalCostRdis = fetcher.hfetch(getTicketCostCosredis(adTicketCost.getTicketId(),date),  adTicketCost.getId().toString(), Double.class);
                //发送钉钉推送
                eventPark.post(new TicketCostExceedCostWarnEvent(this, ticket, val, finalCostRdis, clickRequest.getPosition(),adTicketCost.getId()));
                //开始修改阀值（之前消耗的值+需要提醒的值）下次报警的阀值
                updater.hupdate(getTicketCostConsume(adTicketCost.getTicketId(),date),
                        adTicketCost.getId().toString(), cosWarn + val-val%cosWarn, (int) seconds);
                //修改上次的最后保存的阀值
                updater.hupdate(getTicketCostCosredis(adTicketCost.getTicketId(),date),
                        adTicketCost.getId().toString(), val-val%cosWarn, (int) seconds);
            }
        } catch (Exception e) {
            LOGGER.error("券消耗提醒记录添加失败", e);
        }
    }


    /**
     * 处理广告位-广告券-时间段 消耗控制
     * {@see com.bxm.adsprod.pushable.position.PositionTicketTimesLimitPushable}
     * @param val
     * @param clickRequest
     * @param ticket
     */
    private void processPositionTicketTimesLimit(long val, ClickRequest clickRequest, Ticket ticket) {

        final String position = clickRequest.getPosition();
        final TicketPositionTimesLimit currentHourTicketPositionTimesLimit =
                ticketPositionTimesLimitService.getCurrentHourTicketPositionTimesLimitByTicketAndPosition(ticket.getId().toString(), position);
        if (currentHourTicketPositionTimesLimit == null){
            return;
        }

        final String configId = currentHourTicketPositionTimesLimit.getConfigId().toString();

        final Long currentConsume = counter.incrementByAndGet(
                TicketKeyGenerator.ticketPositionTimesLimitConfigCount(ticket.getId(),configId,position), val, 24 * 60 * 60);

        long limitPrice = Optional.ofNullable(currentHourTicketPositionTimesLimit.getLimitPrice()).orElse(-1L);
        if (limitPrice >= 0 && currentConsume >= limitPrice){
            updater.hupdate(TicketKeyGenerator.ticketPositionTimesLimitConfigClose( ticket.getId(),configId),position ,ticket.getId(), 24*60*60);

            // 按广告位维度存储，这样在查询时时间复杂度低。
            String ticketId = String.valueOf(ticket.getId());
            updater.supdate(TicketKeyGenerator.setTicketClosedPosition(position), 24 * 60 * 60, ticketId);
        }

    }

    private void handleClosed(ClickRequest clickRequest, Ticket ticket) {
        try {
            String positionId = clickRequest.getPosition();
            BigInteger ticketId = ticket.getId();
            PositionGroupRuleSo.Entry config = fetcher.hfetch(TicketKeyGenerator.Filter.getPositionGroup(ticketId), positionId, PositionGroupRuleSo.Entry.class);
            if (null == config) {
                return;
            }
            String groupId = config.getGroupId();
            long limit = config.getLimit();

            Long current = positionGroupService.getBudgetOfToday(ticketId, groupId);
            if (limit >= 0 && current >= limit) {
                // closed
                if (LOGGER.isInfoEnabled()) {
                    LOGGER.info("Closed [{}] at {}, cfg: groupId={},limit={},budget={}",
                            ticketId, groupId, groupId, limit, current);
                }
                updater.supdate(TicketKeyGenerator.Position.getClosedTicket(groupId), 24 * 60 * 60, String.valueOf(ticketId));
            }
        } catch (Exception e) {
            LOGGER.error("handleClosed: ", e);
        }
    }

    @Override
    protected KeyGenerator getKeyGenerator(String uid, BigInteger ticketId) {
        return TicketKeyGenerator.Statistics.getBudgetOfDaily();
    }

}
