package com.bxm.adsprod.counter.ticket.counter;

import com.bxm.adsprod.counter.event.TicketClickEvent;
import com.bxm.adsprod.counter.ticket.AbstractClickCounter;
import com.bxm.adsprod.counter.ticket.HashCounter;
import com.bxm.adsprod.facade.ticket.*;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.math.BigInteger;

/**
 * 广告组频次点击限制
 *
 * @author 阿导
 * @version 1.0
 * @fileName com.bxm.adsprod.counter.ticket.counter.TicketGroupClickCounter.java
 * @CopyRright (c) 2018-万物皆导
 * @created 2018-03-27 14:55:00
 */
@Component
public class TicketGroupDailyClickCounter extends AbstractClickCounter implements HashCounter, EventListener<TicketClickEvent> {

    @Autowired
    private Fetcher fetcher;
    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(TicketClickEvent event) {
        super.consume(event.getRequest(), event.getTicket());
    }

    @Override
    protected long getIncrementValue(ClickRequest clickRequest, Ticket ticket) {
        return clickRequest.isValid() ? 0 : DEFAULT_INCREMENT_VALUE;
    }

    /**
     * 会首先从{@link #getKeyGenerator(String, BigInteger)}获取，如果返回为<code>null</code> 则使用这个返回存储需要的KeyGenerator实现
     *
     * @param uid
     * @param ticketId
     * @return
     * @see #getKeyGenerator(String, BigInteger)
     */
    @Override
    protected KeyGenerator getKeyGenerator(String uid, BigInteger ticketId) {
        return null;
    }

    /**
     * 返回存储需要的KeyGenerator实现，可以覆盖这个方法。
     *
     * @param clickRequest
     * @param ticket
     * @return
     * @see #getKeyGenerator(String, BigInteger)
     */
    @Override
    protected KeyGenerator getKeyGenerator(ClickRequest clickRequest, Ticket ticket) {
        Long groupId = ticket.getGroupId();
        return TicketKeyGenerator.Statistics.getTicketGroupDailyClickKey(groupId);
    }

    /**
     * 返回过期时间，默认为10天
     *
     * @return
     */
    @Override
    protected int getExpireTimeInSeconds() {
        return 24 * 60 * 60;
    }


    @Override
    public <T extends CounterRequest> String getField(T request) {
        ClickRequest click = convertRequest(request);
        return click.getUid();
    }

    /**
     * 是否需要执行increment
     *
     * @param clickRequest
     * @param ticket
     * @return boolean
     * @description
     * @author 阿导
     * @time 2018/3/29
     * @CopyRight 杭州微财科技有限公司
     */
    @Override
    protected boolean isIncrementIfNeeded(ClickRequest clickRequest, Ticket ticket) {
        return null != ticket.getGroupId();
    }

    @Override
    protected void afterIncrement(long val, ClickRequest clickRequest, Ticket ticket) {
        processAdx(val, clickRequest, ticket);

        Long groupId = ticket.getGroupId();
        if (null == groupId) {
            return;
        }

        TicketGroupPushVo vo = fetcher.fetch(TicketKeyGenerator.Filter.getTicketGroup(BigInteger.valueOf(groupId)), TicketGroupPushVo.class);
        if (!vo.isEnabled()) {
            return;
        }
        String uid = clickRequest.getUid();

        Integer clickFrequency = vo.getClickFrequency();
        if (null != clickFrequency && clickFrequency.equals(TicketGroupPushVo.EVERY_DAY)) {
            Integer current = fetcher.hfetch(TicketKeyGenerator.Statistics.getTicketGroupDailyClickKey(groupId), uid, Integer.class);
            Integer clickFrequencyNum = vo.getClickFrequencyNum();
            if (null != current && null != clickFrequencyNum && current >= clickFrequencyNum) {
                // Closed
                updater.supdate(TicketKeyGenerator.User.getClosedGroup(uid), 24 * 60 * 60, String.valueOf(groupId));
            }
        }

    }

    /**
     * 处理adx中普通券的组/频次控制
     * @param val
     * @param clickRequest
     * @param ticket
     */
    protected void processAdx(long val, ClickRequest clickRequest, Ticket ticket) {
        Long groupId = ticket.getGroupId();
        if (null == groupId) {
            return;
        }
        //是否开启广告组
        TicketGroupPushVo vo = fetcher.fetch(TicketKeyGenerator.Filter.getTicketGroup(BigInteger.valueOf(groupId)), TicketGroupPushVo.class);
        if (!vo.isEnabled()) {
            return;
        }
        String uid = clickRequest.getUid();
        //判断用户id是否为空，为空不作处理
        if (StringUtils.isEmpty(uid)) {
            return;
        }
        Integer clickFrequency = vo.getClickFrequency();
        //判断广告组是否按每天控制
        if (null != clickFrequency && clickFrequency.equals(TicketGroupPushVo.EVERY_DAY)) {
            Integer current = fetcher.hfetch(TicketKeyGenerator.Statistics.getTicketGroupDailyClickKey(groupId), uid, Integer.class);
            Integer clickFrequencyNum = vo.getClickFrequencyNum();
            if (null != current && null != clickFrequencyNum && current >= clickFrequencyNum) {
                // Closed
                KeyGenerator key = TicketKeyGenerator.User.getClosedTicketByUid(uid);
                //添加已关闭的券id（redis-val是set集合）
                updater.supdate(key,  24 * 60 * 60, String.valueOf(ticket.getId()));
            }
        }
    }
}
