package com.bxm.adsprod.counter.ticket.counter;

import com.bxm.adsprod.counter.event.TicketClickEvent;
import com.bxm.adsprod.counter.ticket.AbstractClickCounter;
import com.bxm.adsprod.counter.ticket.HashCounter;
import com.bxm.adsprod.facade.ticket.*;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.math.BigInteger;

/**
 * 广告组频次点击限制
 *
 * @author 阿导
 * @version 1.0
 * @fileName com.bxm.adsprod.counter.ticket.counter.TicketGroupClickCounter.java
 * @CopyRright (c) 2018-万物皆导
 * @created 2018-03-27 14:55:00
 */
@Component
public class TicketGroupWeekClickCounter extends AbstractClickCounter implements HashCounter, EventListener<TicketClickEvent> {

    @Autowired
    private Fetcher fetcher;
    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(TicketClickEvent event) {
        super.consume(event.getRequest(), event.getTicket());
    }

    @Override
    protected long getIncrementValue(ClickRequest clickRequest, Ticket ticket) {
        return clickRequest.isValid() ? 0 : DEFAULT_INCREMENT_VALUE;
    }

    @Override
    protected KeyGenerator getKeyGenerator(String uid, BigInteger ticketId) {
        return null;
    }

    /**
     * 返回存储需要的KeyGenerator实现，可以覆盖这个方法。
     *
     * @param clickRequest
     * @param ticket
     * @return
     * @see #getKeyGenerator(String, BigInteger)
     */
    @Override
    protected KeyGenerator getKeyGenerator(ClickRequest clickRequest, Ticket ticket) {
        Long groupId = ticket.getGroupId();
        return TicketKeyGenerator.Statistics.getTicketGroupWeekClickKey(groupId);
    }

    /**
     * 返回HASH数据的键
     *
     * @param uid
     * @param ticketId
     * @return
     */
	@Override
	public <T extends CounterRequest> String getField(T request) {
		ClickRequest click = convertRequest(request);
		return click.getUid();
	}
    /**
     * 返回过期时间，默认为10天
     *
     * @return
     */
    @Override
    protected int getExpireTimeInSeconds() {
        return 24*60*60*7;
    }

    @Override
    protected boolean isIncrementIfNeeded(ClickRequest clickRequest, Ticket ticket) {
        return null != ticket.getGroupId();
    }

    @Override
    protected void afterIncrement(long val, ClickRequest clickRequest, Ticket ticket) {
        processAdx(val, clickRequest, ticket);
        Long groupId = ticket.getGroupId();
        if (null == groupId) {
            return;
        }

        TicketGroupPushVo vo = fetcher.fetch(TicketKeyGenerator.Filter.getTicketGroup(BigInteger.valueOf(groupId)), TicketGroupPushVo.class);
        if (!vo.isEnabled()) {
            return;
        }
        String uid = clickRequest.getUid();

        Integer clickFrequency = vo.getClickFrequency();
        if (null != clickFrequency && clickFrequency.equals(TicketGroupPushVo.EVERY_WEEK)) {
            Integer current = fetcher.hfetch(TicketKeyGenerator.Statistics.getTicketGroupWeekClickKey(groupId), uid, Integer.class);
            Integer clickFrequencyNum = vo.getClickFrequencyNum();
            if (null != current && null != clickFrequencyNum && current >= clickFrequencyNum) {
                // Closed
                updater.supdate(TicketKeyGenerator.User.getClosedGroup(uid), 24 * 60 * 60, String.valueOf(groupId));
            }
        }
    }

    private void processAdx(long val, ClickRequest clickRequest, Ticket ticket) {
        Long groupId = ticket.getGroupId();
        if (null == groupId) {
            return;
        }

        TicketGroupPushVo vo = fetcher.fetch(TicketKeyGenerator.Filter.getTicketGroup(BigInteger.valueOf(groupId)), TicketGroupPushVo.class);
        if (!vo.isEnabled()) {
            return;
        }
        String uid = clickRequest.getUid();
        if (StringUtils.isEmpty(uid)) {
            return;
        }
        Integer clickFrequency = vo.getClickFrequency();
        if (null != clickFrequency && clickFrequency.equals(TicketGroupPushVo.EVERY_WEEK)) {
            Integer current = fetcher.hfetch(TicketKeyGenerator.Statistics.getTicketGroupWeekClickKey(groupId), uid, Integer.class);
            Integer clickFrequencyNum = vo.getClickFrequencyNum();
            if (null != current && null != clickFrequencyNum && current >= clickFrequencyNum) {
                // Closed
                KeyGenerator key = TicketKeyGenerator.User.getClosedTicketByUid(uid);
                updater.supdate(key, 24 * 60 * 60, String.valueOf(ticket.getId()));
            }
        }
    }
}
