package com.bxm.adsprod.counter.ticket.counter;

import com.bxm.adsprod.counter.event.TicketViewEvent;
import com.bxm.adsprod.counter.ticket.AbstractViewCounter;
import com.bxm.adsprod.counter.ticket.HashCounter;
import com.bxm.adsprod.counter.user.UserStatisticsService;
import com.bxm.adsprod.counter.utils.AdsServiceDateHelper;
import com.bxm.adsprod.counter.utils.DateUtils;
import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.adsprod.facade.ticket.CounterRequest;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.ViewRequest;
import com.bxm.adsprod.model.so.rules.TicketTimesRuleSo;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.math.BigInteger;
import java.util.List;
import java.util.Map;

/**
 * <h3>广告券用户每天曝光次数统计</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/08
 */
@Component
public class TicketUserDailyViewCounter extends AbstractViewCounter implements HashCounter, EventListener<TicketViewEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketUserDailyViewCounter.class);
    @Autowired
    private UserStatisticsService userStatisticsService;
    @Autowired
    private Fetcher fetcher;
    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(TicketViewEvent event) {
        super.consume(event.getRequest(), event.getTicket());
    }

    /**
     * 广告券频次控制需求--用户N天可曝光M次，现N的取值范围暂为：1-7天，所以redis超时时间设置为7天
     */
    @Override
    protected int getExpireTimeInSeconds() {
        return TypeHelper.castToInt(DateHelper.getRemainSecondsOfDay(7));
    }

    @Override
    protected KeyGenerator getKeyGenerator(String uid, BigInteger ticketId) {
        return TicketKeyGenerator.Statistics.getViewOfDaily(ticketId);
    }

    @Override
	public <T extends CounterRequest> String getField(T request) {
		ViewRequest view = convertRequest(request);
		return view.getUid();
	}

    @Override
    protected void afterIncrement(long val, ViewRequest request, Ticket ticket) {
        processAdx(val, request, ticket);
        BigInteger ticketId = ticket.getId();
        Map<String, Object> parameters = Maps.newHashMap();
        parameters.put(CachePushableFields.TICKET_ID, ticketId);
        TicketTimesRuleSo rule = fetcher.fetch(TicketKeyGenerator.getTimes(parameters), null, TicketTimesRuleSo.class);
        if (null == rule) {
            return;
        }
        String uid = request.getUid();
        List<TicketTimesRuleSo.Entry> entries = rule.getEntries();
        KeyGenerator keyClosedTickets = TicketKeyGenerator.User.getClosedTickets(uid);
        double score = (double) System.currentTimeMillis();
        for (TicketTimesRuleSo.Entry entry : entries) {
            int cycle = entry.getCycle();
            long max = entry.getCount();
            if (entry.getType() == TicketTimesRuleSo.Entry.TYPE_VIEW) {
                if (userStatisticsService.countViewOfSeveralDays(uid, ticketId, cycle) >= max) {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("Closed [{}] at {}", ticketId, uid);
                    }
                    updater.zupdate(keyClosedTickets, score,
                            String.valueOf(ticketId),
                            24 * 60 * 60);
                }
            }
        }
    }

    private void processAdx(long val, ViewRequest request, Ticket ticket) {
        if (ticket.getType() != Ticket.TYPE_ADX_ACTIVITY &&
                ticket.getType() != Ticket.TYPE_ADX_DIRECT) {
            return;
        }
        BigInteger ticketId = ticket.getId();
        Map<String, Object> parameters = Maps.newHashMap();
        parameters.put(CachePushableFields.TICKET_ID, ticketId);
        TicketTimesRuleSo rule = fetcher.fetch(TicketKeyGenerator.getTimes(parameters), null, TicketTimesRuleSo.class);
        if (null == rule) {
            return;
        }
        String uid = request.getUid();
        if (StringUtils.isEmpty(uid)) {
            return;
        }
        List<TicketTimesRuleSo.Entry> entries = rule.getEntries();
        for (TicketTimesRuleSo.Entry entry : entries) {
            int cycle = entry.getCycle();
            long max = entry.getCount();
            int type = entry.getType();
            String r = cycle + "-" + type + "-" + max;
            if (entry.getType() == TicketTimesRuleSo.Entry.TYPE_VIEW) {
                counterToday(uid, ticketId, cycle, max , r);
                counterSeveralDays(uid, ticketId, cycle, max , r);
            }
        }
    }

    private void counterToday(String uid, BigInteger ticketId, int cycle, long max, String rule) {
        KeyGenerator viewOfDaily = TicketKeyGenerator.Statistics.getViewOfDaily(ticketId);
        long count = counter.hget(viewOfDaily, uid);
        if (count >= max) {
            List<String> dateList = AdsServiceDateHelper.getSeveralDateListLater(cycle);
            for (String d : dateList) {
                int second = cycle * 24 * 3600;
                KeyGenerator key = TicketKeyGenerator.User.getViewClosedTicketByUidV2(d, uid);
                updater.hupdate(key, String.valueOf(ticketId), rule, second);
            }
        }
    }

    private void counterSeveralDays(String uid, BigInteger ticketId, int cycle, long max, String rule) {
        if (userStatisticsService.countViewOfSeveralDays(uid, ticketId, cycle) >= max) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Closed [{}] at {}", ticketId, uid);
            }
            long second = DateUtils.getDayRemainingTime();
            KeyGenerator key = TicketKeyGenerator.User.getViewClosedTicketByUidV2(uid);
            updater.hupdate(key, String.valueOf(ticketId), rule, (int) second);
        }
    }
}
