package com.bxm.adsmedia.common.util;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.codec.digest.DigestUtils;

/**
 * <p>
 * MD5签名和验签 工具类
 * </p>
 *
 * @author zhengwangeng
 * @since 2020-05-18
 */
public class BianXianMaoSignUtil {

    private final static String APP_SECRET = "appSecret";

    private final static String SIGN = "sign";

    private final static String TIMESTAMP = "timestamp";

    //时间戳有效期，分钟
    private final static Long EXPIRES_SECOND = 10L;

    /**
     * 方法描述:将字符串MD5加码 生成32位md5码
     *
     * @param inStr
     * @return
     */
    public static String md5(String inStr) {
        try {
            return DigestUtils.md5Hex(inStr.getBytes("UTF-8"));
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("MD5签名过程中出现错误");
        }
    }

    /**
     * 方法描述:签名字符串
     *
     * @param params    需要签名的参数
     * @param appSecret 签名密钥
     * @return
     */
    public static String sign(HashMap<String, String> params, String appSecret) {
        StringBuilder valueSb = new StringBuilder();
        params.put(APP_SECRET, appSecret);
        // 将参数以参数名的字典升序排序
        Map<String, String> sortParams = new TreeMap<>(params);
        Set<Entry<String, String>> entrys = sortParams.entrySet();
        // 遍历排序的字典,并拼接value1+value2......格式
        for (Entry<String, String> entry : entrys) {
            String value = entry.getValue();
            if (value == null || "".equals(value)) {
                //过滤空值
                continue;
            }
            valueSb.append(value);
        }
        params.remove(APP_SECRET);
        return md5(valueSb.toString());
    }

    /**
     * 方法描述:验证签名
     *
     * @param appSecret 加密秘钥
     * @param request
     * @return
     * @throws Exception
     */
    public static boolean verify(String appSecret, HttpServletRequest request) throws Exception {
        String sign = request.getParameter(SIGN);
        if (sign == null || "".equals(sign)) {
            throw new Exception("请求中必须带签名参数，且不能为空");
        }
        String timestamp = request.getParameter(TIMESTAMP);
        if (timestamp == null || "".equals(timestamp)) {
            throw new Exception("请求中必须带时间戳参数，且不能为空");
        }
        Long second = null;
        try {
            second = (System.currentTimeMillis() - Long.parseLong(timestamp)) / (1000 * 60);
        } catch (Exception e) {
            throw new Exception("时间戳参数必须为13位数字");
        }
        if (second > EXPIRES_SECOND) {
            throw new Exception("timestamp有效期超过十分钟");
        }

        HashMap<String, String> params = new HashMap(10);

        // 获取url参数
        Enumeration<String> enu = request.getParameterNames();
        while (enu.hasMoreElements()) {
            String paramName = enu.nextElement().trim();
            // 拼接参数值字符串并进行utf-8解码，防止中文乱码产生
            params.put(paramName, URLDecoder.decode(request.getParameter(paramName), "UTF-8"));
        }
        //移除签名
        params.remove(SIGN);

        return verify(params, sign, appSecret);
    }


    public static boolean verify(HashMap<String, String> params, String sign, String appSecret) throws Exception {
        if (sign == null || "".equals(sign)) {
            throw new Exception("请求中必须带签名参数，且不能为空");
        }
        params.put(APP_SECRET, appSecret);

        // 将参数以参数名的字典升序排序
        Map<String, String> sortParams = new TreeMap<>(params);
        Set<Entry<String, String>> entrys = sortParams.entrySet();

        // 遍历排序的字典,并拼接value1+value2......格式
        StringBuilder valueSb = new StringBuilder();
        for (Entry<String, String> entry : entrys) {
            String value = entry.getValue();
            if (value == null || "".equals(value)) {
                //过滤空值
                continue;
            }
            valueSb.append(value);
        }

        return sign.equals(md5(valueSb.toString()));
    }

    public static void main(String[] args) throws Exception {
        HashMap<String, String> map = new HashMap<>();

        map.put("date", "2022-01-13");

        //签名
        map.put("appKey", "722928e2c525490aa0b17265dfabd4a3");
        long currentTimeMillis = System.currentTimeMillis();
        currentTimeMillis = 1596676932762L;
        System.out.println(currentTimeMillis);
        //map.put("timestamp", String.valueOf(currentTimeMillis));
        map.put("timestamp", Objects.toString(currentTimeMillis));
        map.put("positionCode", "000480-1");


        String appSecret = "ff1109851ff34deaa0046cfcee1694ce";
        //加密
        System.out.println(sign(map, appSecret));
        //校验
        String sign = "726b03cc2d26ed71df2576bb55959200";
        System.out.println(verify(map, sign, appSecret));
    }

}

