package com.bxm.adsmedia.common.util;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.apache.commons.lang3.StringUtils;

import javax.activation.DataHandler;
import javax.activation.FileDataSource;
import javax.mail.*;
import javax.mail.internet.*;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.util.*;

/**
 * <p>发送电子邮件</p>
 *
 * @author kerry.jiang
 * @date 2019-7-10 17:48
 */
public class EmailSender {

    private static Map<String, Session> sessionMap = new HashMap<>();

    private Session session;//会话
    private Boolean cacheSession = Boolean.TRUE;//是否缓存session
    private Properties props;//会话属性
    private MimeMessage mimeMessage; //邮件对象

    private Email email;//邮件信息

    private EmailSender(Boolean cacheSession, Email email) {
        if(null != cacheSession){
            this.cacheSession = cacheSession;
        }
        this.email = email;
        init();
    }

    private EmailSender(Boolean cacheSession, String host, String username, String password) {
        if(null != cacheSession){
            this.cacheSession = cacheSession;
        }
        this.email = new Email();
        this.email.setAccount(new SenderAccount(host, username, password));
        init();
    }

    private EmailSender(Boolean cacheSession, String host, String username, String password, Integer port, Boolean ssl, String encoding) {
        if(null != cacheSession){
            this.cacheSession = cacheSession;
        }
        this.email = new Email();
        this.email.setAccount(new SenderAccount(host, username, password, port, ssl, encoding));
        init();
    }

    /**
     * <p>初始化</p>
     */
    private void init() {
        if(Boolean.TRUE == cacheSession){
            this.session = getCacheSession();
        }else{
            this.session = getNewSession();
        }
        //用session对象来创建并初始化邮件对象
        this.mimeMessage = new MimeMessage(this.session);
    }

    /**
     * <p>获取会话</p>
     * <p>根据host缓存</p>
     */
    private Session getCacheSession(){
        String key = this.email.getAccount().getHost();
        Session session = sessionMap.get(key);
        if(session == null){
            session = getNewSession();
            sessionMap.put(key, session);
        }
        return session;
    }

    /**
     * <p>获取会话</p>
     * <p>每次获取新的</p>
     */
    private Session getNewSession(){
        if (this.props == null) {
            this.props = System.getProperties();
        }
        SenderAccount account = email.getAccount();
        this.props.put("mail.smtp.host", account.getHost());//邮件服务器主机名
        this.props.put("mail.smtp.port", account.getPort());//邮件服务器端口
        this.props.put("mail.smtp.socketFactory.port", account.getPort());//邮件服务器端口
        if(Boolean.TRUE == account.getSsl()){
            this.props.put("mail.smtp.socketFactory.class", "javax.net.ssl.SSLSocketFactory"); //SSL
        }
        this.props.put("mail.smtp.auth", "false"); //需要身份验证
        this.props.put("mail.debug", "false"); //控制台看到发送邮件的过程
        return Session.getInstance(this.props);
    }

    /**
     * <p>创建对象</p>
     *
     * @param cacheSession 是否缓存会话
     * @param email 邮件信息
     * @return com.bxm.warcar.message.email.EmailSender
     */
    public static EmailSender create(Boolean cacheSession, Email email) throws MessagingException {
        return new EmailSender(cacheSession, email);
    }

    /**
     * <p>创建对象</p>
     *
     * @param host 邮箱服务器地址
     * @param username 用户名
     * @param password 密码
     * @param port 端口
     * @return com.bxm.warcar.message.email.EmailSender
     */
    public static EmailSender create(String host, String username, String password, Integer port) throws MessagingException {
        return new EmailSender(Boolean.TRUE, host, username, password, port, null, null);
    }

    /**
     * <p>创建对象</p>
     *
     * @param host 邮箱服务器地址
     * @param username 用户名
     * @param password 密码
     * @param port 端口
     * @param cacheSession 是否缓存会话
     * @return com.bxm.warcar.message.email.EmailSender
     */
    public static EmailSender create(String host, String username, String password, Integer port, Boolean cacheSession) throws MessagingException {
        return new EmailSender(cacheSession, host, username, password, port, null, null);
    }

    /**
     * <p>创建对象</p>
     *
     * @param host 邮箱服务器地址
     * @param username 用户名
     * @param password 密码
     * @param port 端口
     * @param ssl 支持ssl
     * @param cacheSession 是否缓存会话
     * @return com.bxm.warcar.message.email.EmailSender
     */
    public static EmailSender create(String host, String username, String password, Integer port, Boolean ssl, Boolean cacheSession) throws MessagingException {
        return new EmailSender(cacheSession, host, username, password, port, ssl, null);
    }

    /**
     * <p>创建对象</p>
     *
     * @param host 邮箱服务器地址
     * @param username 用户名
     * @param password 密码
     * @param port 端口
     * @param ssl 支持ssl
     * @param encoding 编码
     * @param cacheSession 是否缓存会话
     * @return com.bxm.warcar.message.email.EmailSender
     */
    public static EmailSender create(String host, String username, String password, Integer port, Boolean ssl, String encoding, Boolean cacheSession) throws MessagingException {
        return new EmailSender(cacheSession, host, username, password, port, ssl, encoding);
    }

    /**
     * <p>设置发件人</p>
     * @param from 发件人
     */
    public EmailSender setFrom(String from) throws MessagingException {
        this.email.setFrom(from);
        return this;
    }

    /**
     * <p>设置收件人</p>
     * @param toList 收件人
     */
    public EmailSender setToList(List<String> toList) {
        this.email.setToList(toList);
        return this;
    }

    /**
     * <p>设置抄送</p>
     * @param ccList 抄送
     */
    public EmailSender setCcList(List<String> ccList) {
        this.email.setCcList(ccList);
        return this;
    }

    /**
     * <p>设置密送</p>
     * @param bccList 密送
     */
    public EmailSender setBccList(List<String> bccList) {
        this.email.setBccList(bccList);
        return this;
    }

    /**
     * <p>设置主题</p>
     * @param subject 主题
     */
    public EmailSender setSubject(String subject) {
        this.email.setSubject(subject);
        return this;
    }

    /**
     * <p>设置内容</p>
     * @param text 文本内容
     */
    public EmailSender setText(String text) {
        this.email.setContent(text);
        this.email.setType(1);
        return this;
    }

    /**
     * <p>设置内容</p>
     * @param html 网页内容
     */
    public EmailSender setHtml(String html, String charset) {
        this.email.setContent(html);
        this.email.setType(2);
        if(charset == null){
            this.email.setCharset("UTF-8");
        }else{
            this.email.setCharset(charset);
        }
        return this;
    }

    /**
     * <p>设置内容图片</p>
     * @param imageMap 内容图片
     */
    public EmailSender setImageMap(Map<String, String> imageMap) {
        this.email.setImageMap(imageMap);
        return this;
    }

    /**
     * <p>设置附件</p>
     * @param attachmentList 附件
     */
    public EmailSender setAttachmentList(List<String> attachmentList) {
        this.email.setAttachmentList(attachmentList);
        return this;
    }

    /**
     * <p>添加收件人</p>
     * @param to 收件人
     */
    public EmailSender addTo(String to) {
        if(this.email.getToList() == null){
            this.email.setToList(new ArrayList<>());
        }
        this.email.getToList().add(to);
        return this;
    }

    /**
     * <p>添加抄送</p>
     * @param cc 抄送
     */
    public EmailSender addCc(String cc) {
        if(this.email.getCcList() == null){
            this.email.setCcList(new ArrayList<>());
        }
        this.email.getCcList().add(cc);
        return this;
    }

    /**
     * <p>添加密送</p>
     * @param bcc 密送
     */
    public EmailSender addBcc(String bcc) {
        if(this.email.getBccList() == null){
            this.email.setBccList(new ArrayList<>());
        }
        this.email.getBccList().add(bcc);
        return this;
    }

    /**
     * <p>添加内容图片</p>
     * @param cid 图片占位ID
     * @param image 图片地址(本地地址,非网络地址)
     */
    public EmailSender addImage(String cid, String image) {
        if(this.email.getImageMap() == null){
            this.email.setImageMap(new HashMap<>());
        }
        this.email.getImageMap().put(cid, image);
        return this;
    }

    /**
     * <p>添加附件</p>
     * @param attachment 附件
     */
    public EmailSender addAttachment(String attachment) {
        if(this.email.getAttachmentList() == null){
            this.email.setAttachmentList(new ArrayList<>());
        }
        this.email.getAttachmentList().add(attachment);
        return this;
    }

    /**
     * 发送邮件
     */
    public boolean send() throws Exception {
        //预备
        prepare();

        SenderAccount account = this.email.getAccount();
        Transport transport = this.session.getTransport("smtp");
        // 连接邮件服务器并进行身份验证
        transport.connect(account.getHost(), account.getUsername(), account.getPassword());
        // 发送邮件
        transport.sendMessage(this.mimeMessage, this.mimeMessage.getAllRecipients());
        transport.close();
        return Boolean.TRUE;
    }

    /**
     * <p>预备</p>
     */
    private void prepare() throws MessagingException, UnsupportedEncodingException {
        SenderAccount account = this.email.getAccount();
        //附带组件
        Multipart multipart = new MimeMultipart();
        this.mimeMessage.setContent(multipart);
        //发件人
        this.mimeMessage.setFrom(new InternetAddress(account.getUsername(), encode(this.email.getFrom())));
        //收件人
        if(CollectionUtils.isNotEmpty(this.email.getToList())){
            String to = StringUtils.join(this.email.getToList(), ",");
            this.mimeMessage.setRecipients(Message.RecipientType.TO, InternetAddress.parse(to));
        }
        //抄送
        if(CollectionUtils.isNotEmpty(this.email.getCcList())){
            String cc = StringUtils.join(this.email.getCcList(), ",");
            this.mimeMessage.setRecipients(Message.RecipientType.CC, InternetAddress.parse(cc));
        }
        //密送
        if(CollectionUtils.isNotEmpty(this.email.getBccList())){
            String bcc = StringUtils.join(this.email.getBccList(), ",");
            this.mimeMessage.setRecipients(Message.RecipientType.BCC, InternetAddress.parse(bcc));
        }
        //主题
        this.mimeMessage.setSubject(this.email.getSubject());
        //邮件内容
        MimeBodyPart bodyPart = new MimeBodyPart();
        if(2 == this.email.getType()){//html
            String charset = this.email.charset == null ? "UTF-8" : this.email.charset;
            String content = StringUtils.join("<meta http-equiv=Content-Type content=text/html;charset=", charset,">", email.getContent());
            bodyPart.setContent(content, StringUtils.join("text/html;charset=", charset));
        }else{
            bodyPart.setText(this.email.getContent());
        }
        multipart.addBodyPart(bodyPart);
        //图片
        Map<String, String> imageMap = this.email.getImageMap();
        if(MapUtils.isNotEmpty(imageMap)){
            Set<String> keySet = imageMap.keySet();
            for (String cid : keySet){
                String image = imageMap.get(cid);
                bodyPart = new MimeBodyPart();
                // 读取本地文件
                DataHandler dh = new DataHandler(new FileDataSource(image));
                // 将图片数据添加到"节点"
                bodyPart.setDataHandler(dh);
                // 为"节点"设置一个唯一编号（在文本"节点"将引用该ID）
                bodyPart.setContentID(cid);
                multipart.addBodyPart(bodyPart);// 添加附件
            }
        }

        //附件
        List<String> attachmentList = this.email.getAttachmentList();
        if (CollectionUtils.isNotEmpty(attachmentList)) {
            for (String attachment : attachmentList){
                bodyPart = new MimeBodyPart();
                DataHandler dh = new DataHandler(new FileDataSource(attachment));
                bodyPart.setDataHandler(dh);
                bodyPart.setFileName(encode(dh.getName())); //解决附件名称乱码
                multipart.addBodyPart(bodyPart);// 添加附件
            }
        }
    }

    /**
     * 编码,解决乱码
     */
    private String encode(String source) throws UnsupportedEncodingException {
        return MimeUtility.encodeText(source);
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this.email, ToStringStyle.SHORT_PREFIX_STYLE);
    }

    /**
     * <p>电子邮件</p>
     *
     * @author kerry.jiang
     * @date 2019-7-10 15:49
     */
    public static class Email implements Serializable {

        private static final long serialVersionUID = 7953257301233581347L;

        private SenderAccount account;//发送者帐号
        private String from;//发送人(收件人看到的信息)
        private String subject;//邮件主题
        private String content;//邮件内容
        private Integer type = 1;//内容类型,1:文本内容,2:网页内容
        private String charset;//网页编码
        private List<String> toList;//收件人列表
        private List<String> ccList;//抄送列表
        private List<String> bccList;//密送列表
        private Map<String, String> imageMap;//内容图片(本机地址,非网络地址)
        private List<String> attachmentList;//附件列表(本机地址,非网络地址)

        public Email() {
        }

        public Email(SenderAccount account, String from, String subject, String content, String to) {
            this.account = account;
            this.from = from;
            this.subject = subject;
            this.content = content;
            this.toList = new ArrayList(Arrays.asList(to));
        }

        public Email(SenderAccount account, String from, String subject, String content, List<String> toList) {
            this.account = account;
            this.from = from;
            this.subject = subject;
            this.content = content;
            this.toList = toList;
        }

        public Email(String from, String subject, String content, String to) {
            this.from = from;
            this.subject = subject;
            this.content = content;
            this.toList = new ArrayList(Arrays.asList(to));
        }

        public Email(String from, String subject, String content, List<String> toList) {
            this.from = from;
            this.subject = subject;
            this.content = content;
            this.toList = toList;
        }

        public SenderAccount getAccount() {
            return account;
        }

        public void setAccount(SenderAccount account) {
            this.account = account;
        }

        public String getFrom() {
            return from;
        }

        public void setFrom(String from) {
            this.from = from;
        }

        public String getSubject() {
            return subject;
        }

        public void setSubject(String subject) {
            this.subject = subject;
        }

        public String getContent() {
            return content;
        }

        public void setContent(String content) {
            this.content = content;
        }

        public List<String> getToList() {
            return toList;
        }

        public Integer getType() {
            return type;
        }

        public void setType(Integer type) {
            this.type = type;
        }

        public String getCharset() {
            return charset;
        }

        public void setCharset(String charset) {
            this.charset = charset;
        }

        public void setToList(List<String> toList) {
            this.toList = toList;
        }

        public List<String> getCcList() {
            return ccList;
        }

        public void setCcList(List<String> ccList) {
            this.ccList = ccList;
        }

        public List<String> getBccList() {
            return bccList;
        }

        public void setBccList(List<String> bccList) {
            this.bccList = bccList;
        }

        public Map<String, String> getImageMap() {
            return imageMap;
        }

        public void setImageMap(Map<String, String> imageMap) {
            this.imageMap = imageMap;
        }

        public List<String> getAttachmentList() {
            return attachmentList;
        }

        public void setAttachmentList(List<String> attachmentList) {
            this.attachmentList = attachmentList;
        }

    }

    /**
     * <p>发送者帐号</p>
     */
    public static class SenderAccount implements Serializable {

        private static final long serialVersionUID = 7953257301233581347L;

        private String host;//邮箱服务器地址
        private String username;//用户名
        private String password;//密码
        private Integer port = 25;//端口
        private Boolean ssl = Boolean.TRUE;//是否SSL
        private String encoding = "UTF-8";//字符编码

        public SenderAccount() {
        }

        public SenderAccount(String host, String username, String password) {
            this.host = host;
            this.username = username;
            this.password = password;
        }

        public SenderAccount(String host, String username, String password, Integer port) {
            this.host = host;
            this.username = username;
            this.password = password;
            if(port != null){
                this.port = port;
            }
        }

        public SenderAccount(String host, String username, String password, Integer port, Boolean ssl) {
            this.host = host;
            this.username = username;
            this.password = password;
            if(port != null){
                this.port = port;
            }
            if(ssl != null){
                this.ssl = ssl;
            }
        }

        public SenderAccount(String host, String username, String password, Integer port, Boolean ssl, String encoding) {
            this.host = host;
            this.username = username;
            this.password = password;
            if(port != null){
                this.port = port;
            }
            if(ssl != null){
                this.ssl = ssl;
            }
            if(encoding != null){
                this.encoding = encoding;
            }
        }

        public String getHost() {
            return host;
        }

        public void setHost(String host) {
            this.host = host;
        }

        public String getUsername() {
            return username;
        }

        public void setUsername(String username) {
            this.username = username;
        }

        public String getPassword() {
            return password;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        public Integer getPort() {
            return port;
        }

        public void setPort(Integer port) {
            this.port = port;
        }

        public Boolean getSsl() {
            return ssl;
        }

        public void setSsl(Boolean ssl) {
            this.ssl = ssl;
        }

        public String getEncoding() {
            return encoding;
        }

        public void setEncoding(String encoding) {
            this.encoding = encoding;
        }
    }
}
