package com.bxm.adsprod.counter.ticket;

import com.bxm.adsprod.common.message.AbstractMessageListener;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.TypeHelper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.math.BigInteger;

/**
 * <h3>抽象的统计器</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/11
 */
public abstract class AbstractCounter<REQUEST, RETURNING> extends AbstractMessageListener {

    public static final long DEFAULT_INCREMENT_VALUE = 1;

    @Autowired
    @Qualifier("jedisCounter")
    protected Counter counter;

    /**
     * 转换Request对象
     * @param request
     * @return
     */
    protected abstract REQUEST convertRequest(Object request);

    /**
     * 转换Returning对象
     * @param returning
     * @return
     */
    protected abstract RETURNING convertReturning(Object returning);

    /**
     * 返回存储需要的KeyGenerator实现
     * @param uid
     * @param ticketId
     * @return
     */
    protected abstract KeyGenerator getKeyGenerator(String uid, BigInteger ticketId);

    /**
     * 返回增量值
     * @param request
     * @param returning
     * @return
     */
    protected abstract long getIncrementValue(REQUEST request, RETURNING returning);

    /**
     * 返回UID
     * @param request
     * @param returning
     * @return
     */
    protected abstract String getUid(REQUEST request, RETURNING returning);

    /**
     * 返回广告券ID
     * @param request
     * @param returning
     * @return
     */
    protected abstract BigInteger getTicketId(REQUEST request, RETURNING returning);

    /**
     * 返回过期时间，默认为10天
     * @return
     */
    protected int getExpireTimeInSeconds() {
        return TypeHelper.castToInt(DateHelper.getRemainSecondsOfDay(10));
    }

    /**
     * 处理完成之后需要执行的自定义方法
     * @param val 处理完成之后的值
     * @param request
     * @param returning
     */
    protected void afterIncrement(long val, REQUEST request, RETURNING returning) {}

    @Override
    protected final void consume(Message message, Object request, Object returning) {
        REQUEST r = convertRequest(request);
        RETURNING rtn = convertReturning(returning);
        long incrementValue = getIncrementValue(r, rtn);
        String uid = getUid(r, rtn);
        BigInteger ticketId = getTicketId(r, rtn);
        Long rst = increment(uid, ticketId, incrementValue);
        afterIncrement(rst, r, rtn);
    }

    protected final Long increment(String uid, BigInteger ticketId, long inc) {
        if (this instanceof HashCounter) {
            String field = ((HashCounter) this).getField(uid, ticketId);
            return counter.hincrementByAndGet(getKeyGenerator(uid, ticketId), field, inc, getExpireTimeInSeconds());
        }
        else {
            return counter.incrementByAndGet(getKeyGenerator(uid, ticketId), inc, getExpireTimeInSeconds());
        }
    }
}
