package com.bxm.adsprod.counter.ticket.counter;

import com.bxm.adsprod.common.eventbus.EventPark;
import com.bxm.adsprod.counter.configure.Configuration;
import com.bxm.adsprod.counter.event.BudgetNotEnoughOfDailyEvent;
import com.bxm.adsprod.counter.ticket.AbstractViewCounter;
import com.bxm.adsprod.counter.ticket.HashCounter;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.ViewRequest;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.mq.Consumer;
import com.bxm.warcar.mq.autoconfigure.alions.AlionsConfiguration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Component;

import java.math.BigInteger;

/**
 * <h3>CPA广告券日预算统计器</h3>
 * <p>
 *     CPA 每一次预算触发是：曝光，因此它消费的是广告券曝光主题，并且在内部需要对CPC 的广告券计数增量修改为：0
 * </p>
 * <p>当日预算不足时，会对该广告券的状态更新为暂停</p>
 *
 * @author allen
 * @since V1.0.0 2017/12/11
 */
@Component
@EnableConfigurationProperties({AlionsConfiguration.class, Configuration.class})
public class TicketForCpaDailyBudgetCounter extends AbstractViewCounter implements HashCounter {

    @Autowired
    private AlionsConfiguration configuration;
    @Autowired
    private Configuration local;
    @Autowired
    private EventPark eventPark;

    @Override
    public String getField(String uid, BigInteger ticketId) {
        return String.valueOf(ticketId);
    }

    @Override
    protected long getIncrementValue(ViewRequest ticketRequest, Ticket ticket) {
        return ticket.isCPC() ? 0 : DEFAULT_INCREMENT_VALUE;
    }

    @Override
    protected KeyGenerator getKeyGenerator(String uid, BigInteger ticketId) {
        return TicketKeyGenerator.Statistics.getBudgetOfDaily();
    }

    @Override
    protected void afterIncrement(long val, ViewRequest viewRequest, Ticket ticket) {
        // val = budgetOfToday
        if (val >= ticket.getBudgetDaily()) {
            // Post message to event park.
            eventPark.post(new BudgetNotEnoughOfDailyEvent(this, ticket));
        }
    }

    @Bean
    public Consumer ticketCpaDailyBudgetCounterConsumer() {
        return createConsumer(local.getConsumer().getTicketCpaDailyBudget(),
                configuration.getAccessKey(), configuration.getSecretKey());
    }
}
