package com.bxm.adsprod.service.advertiser;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.adsprod.common.utils.PathUtils;
import com.bxm.adsprod.facade.advertiser.Advertiser;
import com.bxm.adsprod.facade.advertiser.AdvertiserService;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.warcar.cache.DataExtractor;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.LifeCycle;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.BooleanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.List;

/**
 * <h3>Advertiser service initializer</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/25
 */
@Configuration
@EnableConfigurationProperties(AdvertiserConfiguration.class)
public class AdvertiserInitializer extends LifeCycle {
    private static final Logger LOGGER = LoggerFactory.getLogger(AdvertiserServiceImpl.class);

    @Autowired
    private AdvertiserConfiguration configuration;
    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;
    @Autowired
    private AdvertiserService advertiserService;

    @Override
    @PostConstruct
    protected void doInit() {
        if (isInitialized0()) {
            return;
        }
        // 初始化广告主余额
        List<Advertiser> advertisers = getAdvertisers();
        if (CollectionUtils.isNotEmpty(advertisers)) {
            for (Advertiser advertiser : advertisers) {
                BigInteger advertiserId = advertiser.getId();
                // 如果缓存中不存在，那么执行添加余额操作并返回。
                fetcher.hfetch(TicketKeyGenerator.Advertiser.getBalance(), String.valueOf(advertiserId), new DataExtractor<Long>() {
                    @Override
                    public Long extract() {
                        double incValue = advertiser.getBalance().doubleValue();
                        Long balance = advertiserService.incrementBalance(advertiserId, new BigDecimal(incValue).multiply(new BigDecimal(1000)).longValue());
                        if (LOGGER.isDebugEnabled()) {
                            LOGGER.debug("Initializing advertiser {} balance: {}...", advertiserId, balance);
                        }
                        return balance;
                    }
                }, Long.class);
            }
            finishInitialized();
        }
    }

    @Override
    @PreDestroy
    protected void doDestroy() {

    }

    private boolean isInitialized0() {
        Boolean bool = fetcher.fetch(getInitializedKeyGenerator(), null, Boolean.class);
        return BooleanUtils.isTrue(bool);
    }

    private void finishInitialized() {
        updater.update(getInitializedKeyGenerator(), true);
    }

    private KeyGenerator getInitializedKeyGenerator() {
        return new KeyGenerator() {
            @Override
            public String generateKey() {
                return KeyBuilder.build("AD", "ADVERTISER", "INITIALIZED");
            }
        };
    }

    private List<Advertiser> getAdvertisers() {
        RestTemplate restTemplate = new RestTemplateBuilder().build();
        String url = PathUtils.appendToHost(configuration.getHost(), "getList");
        ResponseEntity<String> entity = restTemplate.getForEntity(url, String.class);
        HttpStatus statusCode = entity.getStatusCode();
        HttpHeaders headers = entity.getHeaders();
        String body = entity.getBody();
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Remote address {} responsed: \nstatusCode: {}\nheaders: {}", url, statusCode, headers);
        }
        JSONObject jsonObject = JSONObject.parseObject(body);
        JSONArray list = jsonObject.getJSONArray("returnValue");
        return list.toJavaList(Advertiser.class);
    }
}
