package com.bxm.adsprod.service.commons.message;

import com.bxm.adsprod.common.message.MessageBody;
import com.bxm.adsprod.common.utils.NamedThreadFactory;
import com.bxm.adsprod.service.commons.configure.AdsProdConfiguration;
import com.bxm.adsprod.service.commons.configure.TopicConfiguration;
import com.bxm.adsprod.service.commons.message.annotation.Messaging;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.mq.Producer;
import com.bxm.warcar.mq.SendResult;
import com.bxm.warcar.utils.JsonHelper;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import javax.annotation.PreDestroy;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.concurrent.*;

/**
 * <h3>消息切面，实现了后置通知。</h3>
 * <p>当被拦截的方法执行完成后，会将请求参数和返回对象构造成一个{@link MessageBody}对象，然后对其序列化后发送到消息队列。</p>
 *
 * @see MessageBody
 * @see com.bxm.adsprod.service.commons.message.annotation.Messaging
 *
 * @author allen
 * @since V1.0.0 2017/12/08
 */
@Aspect
@Component
@EnableConfigurationProperties(AdsProdConfiguration.class)
public class MessageAspect {

    private static final Logger LOGGER = LoggerFactory.getLogger(MessageAspect.class);

    private final ExecutorService pool;
    private final Producer producer;
    private final AdsProdConfiguration configuration;

    @Autowired
    public MessageAspect(@Qualifier("alionsProducer") Producer producer, AdsProdConfiguration configuration) {
        this.producer = producer;
        this.configuration = configuration;
        this.pool = new ThreadPoolExecutor(configuration.getMessage().getThreads(),
                configuration.getMessage().getThreads(), 0, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(), new NamedThreadFactory("MsgProducer"));
    }

    @PreDestroy
    public void destroy() {
        pool.shutdown();
    }

    @AfterReturning(pointcut = "@annotation(com.bxm.adsprod.service.commons.message.annotation.Messaging)", returning = "returning")
    public void doAfterReturning(JoinPoint point, Object returning) {
        // 是否需要生产消息
        if (!configuration.getMessage().isProduce()) {
            return;
        }

        Object[] args = point.getArgs();

        Method method = getMethod(point);
        if (null == method) {
            return;
        }
        if (null == returning) {
            return;
        }

        Messaging annotation = method.getAnnotation(Messaging.class);

        if (null == annotation) {
            return;
        }
        if (args.length > 1) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("Not supported multi arguments of Method: {}", method);
            }
            return;
        }

        String topic = getTopic(annotation);
        if (StringUtils.isNotBlank(topic)) {
            try {
                pool.submit(new MessageSendProcessor(producer, topic, returning, args[0], annotation));
            } catch (RejectedExecutionException e) {
                if (LOGGER.isErrorEnabled()) {
                    LOGGER.error("Rejected: topic={}, arg={}, returning={}",
                            topic, args[0], returning);
                }
            }
        }
    }

    private String getTopic(Messaging annotation) {
        try {
            String topic = annotation.value();
            if (StringUtils.isBlank(topic)) {
                TopicConfiguration topicConfiguration = configuration.getMessage().getTopic();
                topic = BeanUtils.getProperty(topicConfiguration, annotation.topicFieldNameOfConfiguration());
            }
            return topic;
        } catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("getTopic: ", e);
            }
            return null;
        }
    }

    private Method getMethod(JoinPoint point) {
        MethodSignature methodSignature = (MethodSignature) point.getSignature();

        Class<?> targetClass = point.getTarget().getClass();
        try {
            return targetClass.getMethod(methodSignature.getName(), methodSignature.getParameterTypes());
        } catch (NoSuchMethodException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("getMethod:", e);
            }
            return null;
        }
    }

    static class MessageSendProcessor implements Runnable {

        private final Producer producer;
        private final String topic;
        private final Object returning;
        private final Object arg;
        private final Messaging annotation;

        MessageSendProcessor(Producer producer, String topic, Object returning, Object arg, Messaging annotation) {
            this.producer = producer;
            this.topic = topic;
            this.returning = returning;
            this.arg = arg;
            this.annotation = annotation;
        }

        @Override
        public void run() {
            sendMessage(returning, arg, annotation);
        }

        private byte[] serialize(Object body) {
            return JsonHelper.convert2bytes(body);
        }

        private void sendMessage(Object returning, Object request, Messaging annotation) {
            String tags = annotation.tags();
            int flag = annotation.flag();
            int delayTimeLevel = annotation.delayTimeLevel();

            MessageBody body = new MessageBody(request, returning);

            Message message = new Message(topic, tags, flag, serialize(body));
            message.setDelayTimeLevel(delayTimeLevel);

            try {
                SendResult result = producer.send(message);
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Message send successful. {}", result.getMsgId());
                }
            } catch (Exception e) {
                // TODO try again?
                if (LOGGER.isErrorEnabled()) {
                    LOGGER.error("Message send failed! ", e);
                }
            }
        }
    }
}
