package com.bxm.adsprod.service.media.pushable;

import com.bxm.adsprod.common.pushable.annotation.CachePush;
import com.bxm.adsprod.convert.media.PositionConvert;
import com.bxm.adsprod.facade.media.PositionOfRules;
import com.bxm.adsprod.facade.rules.Rule;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.model.so.rules.PositionDomainRegionRuleSo;
import com.bxm.adsprod.model.so.rules.PositionTagRegionRuleSo;
import com.bxm.adsprod.service.ticket.filter.PositionDomainRegionInterceptor;
import com.bxm.adsprod.service.ticket.filter.PositionTagRegionInterceptor;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Converter;
import com.bxm.warcar.cache.push.HashUpdating;
import com.bxm.warcar.cache.push.JSONObjectPushable;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <h3>广告位数据推送实现</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/22
 */
@Component
@CachePush("POSITION")
public class PositionPushable extends JSONObjectPushable<PositionOfRules> implements HashUpdating<PositionOfRules> {

    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Autowired
    @Qualifier("positionDomainRegionPushable")
    private Pushable positionDomainRegionPushable;
    @Autowired
    @Qualifier("positionTagRegionPushable")
    private Pushable positionTagRegionPushable;

    public PositionPushable() {
        super(new Converter<PositionOfRules>() {
            @Override
            public Object convert(PositionOfRules object) {
                return PositionConvert.of(object);
            }
        });
    }

    @Override
    protected Class<PositionOfRules> getClsType() {
        return PositionOfRules.class;
    }

    @Override
    protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
        return TicketKeyGenerator.Media.getPositions();
    }

    @Override
    protected Updater getUpdater() {
        return updater;
    }

    @Override
    public String getField(Map<String, Object> parameters, PositionOfRules object) {
        return object.getId();
    }

    @Override
    protected void beforeUpdate(Map<String, Object> parameters, byte[] data, PositionOfRules object) {
        updater.hremove(getKeyGenerator(parameters), getField(parameters, object));

        Rule domainRegion = object.getDomainRegion();
        positionDomainRegionPushable.push(parameters, null == domainRegion ? null : JsonHelper.convert2bytes(domainRegion));

        Rule tagRegion = object.getTagRegion();
        positionTagRegionPushable.push(parameters, null == tagRegion ? null : JsonHelper.convert2bytes(tagRegion));
    }

    @Bean
    public Pushable positionDomainRegionPushable() {
        return new JSONObjectPushable<PositionDomainRegionRuleSo>(new Converter<PositionDomainRegionRuleSo>() {
            @Override
            public Object convert(PositionDomainRegionRuleSo object) {
                if (null == object) {
                    return null;
                }
                Set<String> filter = Sets.newHashSet();
                List<PositionDomainRegionRuleSo.Entry> entries = object.getEntries();
                for (PositionDomainRegionRuleSo.Entry entry : entries) {
                    filter.add(entry.getRegionCode());
                }
                return filter;
            }
        }) {

            @Override
            protected Class<PositionDomainRegionRuleSo> getClsType() {
                return PositionDomainRegionRuleSo.class;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                return PositionDomainRegionInterceptor.keyGenerator(parameters);
            }

            @Override
            protected Updater getUpdater() {
                return updater;
            }
        };
    }

    @Bean
    public Pushable positionTagRegionPushable() {
        return new JSONObjectPushable<PositionTagRegionRuleSo>(new Converter<PositionTagRegionRuleSo>() {
            @Override
            public Object convert(PositionTagRegionRuleSo object) {
                if (null == object) {
                    return null;
                }
                Set<String> filter = Sets.newHashSet();
                List<PositionTagRegionRuleSo.Entry> entries = object.getEntries();
                for (PositionTagRegionRuleSo.Entry entry : entries) {
                    filter.add(entry.getRegionCode());
                }
                return filter;
            }
        }) {

            @Override
            protected Class<PositionTagRegionRuleSo> getClsType() {
                return PositionTagRegionRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                return PositionTagRegionInterceptor.keyGenerator(parameters);
            }
        };
    }
}
