package com.bxm.adsprod.service.ticket;

import com.alibaba.dubbo.config.annotation.Service;
import com.bxm.adsprod.common.interceptor.MutableInvocation;
import com.bxm.adsprod.common.utils.UUIDHelper;
import com.bxm.adsprod.facade.media.Position;
import com.bxm.adsprod.facade.ticket.*;
import com.bxm.adsprod.integration.adsmanager.AdsmanagerService;
import com.bxm.adsprod.service.commons.TicketInterceptorInvocation;
import com.bxm.adsprod.service.commons.message.annotation.Messaging;
import com.bxm.adsprod.service.media.PositionService;
import com.bxm.adsprod.service.ticket.decorator.DecoratorRequestModel;
import com.bxm.adsprod.service.ticket.decorator.TicketDecoratorInterceptorChain;
import com.bxm.adsprod.service.ticket.filter.FilterRequestModel;
import com.bxm.adsprod.service.ticket.filter.TicketFilterInterceptorChain;
import com.bxm.adsprod.service.user.UserService;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.ip.IP;
import com.bxm.warcar.ip.IpLibrary;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.math.BigInteger;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author allen
 * @date 2017-12-05
 */
@Service(version = "1.0.0", owner = "allen")
public class TicketServiceImpl implements TicketService {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketServiceImpl.class);

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired
    private TicketFilterInterceptorChain ticketFilterInterceptorChain;
    @Autowired
    private TicketDecoratorInterceptorChain ticketDecoratorInterceptorChain;
    @Autowired
    private TicketWeightService ticketWeightService;
    @Autowired
    @Qualifier("userServiceImpl")
    private UserService userService;
    @Autowired
    private PositionService positionService;
    @Autowired
    @Qualifier("ipIpNetIpLibrary")
    private IpLibrary ipLibrary;
    @Autowired
    private AdsmanagerService adsmanagerService;

    @Override
    public Ticket get(TicketRequest ticketRequest) {
        List<Ticket> tickets = getValidTickets();
        if (CollectionUtils.isEmpty(tickets)) {
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("没有任何可发的广告券");
            }
            return null;
        }
        String uid = ticketRequest.getUid();
        String ip = ticketRequest.getIp();
        IP iprst = getIp(ip);

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("分析请求来源：{}，结果：{}", ip, iprst);
        }

        // 对所有广告券进行装饰
        doDecorate(tickets, iprst);

        // 区分出常规券、备用券
        List<Ticket> normals = Lists.newArrayList();
        List<Ticket> standbys = Lists.newArrayList();
        for (Ticket ticket : tickets) {
            if (ticket.isNormalType()) {
                normals.add(ticket);
            }
            else {
                standbys.add(ticket);
            }
        }

        // set position
        String positionId = ticketRequest.getPosition();
        Position position = positionService.get(positionId);
        if (null == position) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("非法/无效的广告位: {}", positionId);
            }
            return null;
        }

        // 对常规广告券进行过滤
        doFilter(ticketRequest, normals, position, iprst);

        if (CollectionUtils.isEmpty(normals)) {
            if (CollectionUtils.isEmpty(standbys)) {
                return null;
            }

            // 常规券全部被过滤了，随机一张备用券
            Ticket standby = standbys.get(RandomUtils.nextInt(standbys.size()));
            return returnTicket(standby);
        }
        Ticket lastAcquiredTicket = userService.getLastAcquiredTicket(uid);
        Ticket best = ticketWeightService.getBest(normals, uid, lastAcquiredTicket);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("根据权重最终获得广告券：{}", best);
        }
        return returnTicket(best);
    }

    @Override
    @Messaging(topicFieldNameOfConfiguration = "view")
    public Ticket view(ViewRequest viewRequest) {
        BigInteger ticketId = viewRequest.getTicketId();
        Ticket ticket = get(ticketId);

        if (null == ticket) {
            return null;
        }
        ticket.setTime(new Date());
        return ticket;
    }

    @Override
    @Messaging(topicFieldNameOfConfiguration = "click")
    public Ticket click(ClickRequest clickRequest) {
        BigInteger ticketId = clickRequest.getTicketId();

        Ticket ticket = get(ticketId);
        if (null == ticket) {
            // 兼容老系统的广告券点击，不往外抛异常
            return null;
//            throw new IllegalTicketException();
        }
        Ticket lastAcquiredTicket = userService.getLastAcquiredTicket(clickRequest.getUid());
        if (null == lastAcquiredTicket || ! ticket.getId().equals(lastAcquiredTicket.getId())) {
            throw new IllegalTicketException();
        }
        ticket.setTime(new Date());
        return ticket;
    }

    @Override
    public Ticket get(BigInteger ticketId) {
        return fetcher.hfetch(TicketKeyGenerator.getAllTickets(), String.valueOf(ticketId), null, Ticket.class);
    }

    @Override
    public List<Ticket> getAllTickets() {
        return getAllTicketsForList();
    }

    @Override
    public boolean updateTicketStatus(BigInteger ticketId, byte status, int reason) {
        return adsmanagerService.updateAdTicket(ticketId, status, reason);
    }


    private Ticket returnTicket(Ticket ticket) {
        ticket.setTime(new Date());
        ticket.setBillid(UUIDHelper.generate());
        return ticket;
    }

    private void doFilter(TicketRequest ticketRequest, List<Ticket> normals, Position position, IP iprst) {
        FilterRequestModel requestModel = new FilterRequestModel();
        // 只对常规的券进行过滤
        requestModel.setTickets(normals);
        requestModel.setUid(ticketRequest.getUid());
        requestModel.setImei(ticketRequest.getImei());
        requestModel.setOs(ticketRequest.getOs());
        requestModel.setIp(ticketRequest.getIp());
        requestModel.setApp(ticketRequest.getApp());
        requestModel.setIprst(iprst);
        requestModel.setCity(iprst.getCity());
        requestModel.setPosition(position);

        MutableInvocation invocation = new TicketInterceptorInvocation();
        invocation.setRequestModel(requestModel);

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("开始对常规广告券进行规则过滤，共 {} 张。", normals.size());
        }

        ticketFilterInterceptorChain.intercept(invocation);

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("常规广告券过滤完成，共 {} 张。", normals.size());
        }
    }

    private void doDecorate(List<Ticket> tickets, IP iprst) {
        DecoratorRequestModel decoratorRequestModel = new DecoratorRequestModel();
        decoratorRequestModel.setIprst(iprst);
        decoratorRequestModel.setTickets(tickets);
        MutableInvocation invocation = new TicketInterceptorInvocation();
        invocation.setRequestModel(decoratorRequestModel);
        ticketDecoratorInterceptorChain.intercept(invocation);
    }

    private IP getIp(String ip) {
        IP iprst = ipLibrary.find(ip);
        if (null == iprst) {
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("无法获取到IP：{} 的区域信息", ip);
            }
            iprst = IP.createDefault();
        }
        return iprst;
    }

    /**
     * 获取所有可用的广告券
     *
     * @return 所有可用的广告券
     */
    private List<Ticket> getValidTickets() {
        return getAllTicketsForList(true);
    }

    private List<Ticket> getAllTicketsForList() {
        return getAllTicketsForList(false);
    }

    private List<Ticket> getAllTicketsForList(boolean excludeNotOpenStatus) {
        return Lists.newArrayList(getAllTicketsForMap(excludeNotOpenStatus).values());
    }

    private Map<BigInteger, Ticket> getAllTicketsForMap(boolean excludeNotOpenStatus) {
        Map<String, Ticket> hfetchall = fetcher.hfetchall(TicketKeyGenerator.getAllTickets(), null, Ticket.class);
        if (MapUtils.isEmpty(hfetchall)) {
            return Maps.newHashMap();
        }
        Map<BigInteger, Ticket> result = Maps.newHashMap();
        Set<Map.Entry<String, Ticket>> entries = hfetchall.entrySet();
        for (Map.Entry<String, Ticket> entry : entries) {
            Ticket value = entry.getValue();
            boolean available = !excludeNotOpenStatus || (value.isAvailableForStatus() && value.isAvailableForAssets());
            if (available) {
                result.put(TypeHelper.castToBigInteger(entry.getKey()), value);
            }
        }
        return result;
    }
}
