package com.bxm.adsprod.service.ticket;

import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.TicketWithWeight;
import com.bxm.adsprod.service.user.UserStatisticsService;
import com.bxm.warcar.cache.Fetcher;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Collections;
import java.util.List;

/**
 * 广告券权重服务
 *
 * @author allen
 * @since V1.0.0 2017/12/07
 */
@org.springframework.stereotype.Service
public class TicketWeightServiceImpl implements TicketWeightService {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketWeightServiceImpl.class);
    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired
    private UserStatisticsService userStatisticsService;

    @Override
    public Ticket getBest(List<Ticket> tickets, String uid, Ticket last) {
        List<TicketWithWeight> rst = Lists.newArrayList();
        for (Ticket ticket : tickets) {
            BigInteger ticketId = ticket.getId();
            TicketWithWeight ticketWithWeight = fetcher.fetch(TicketKeyGenerator.Weight.getWeight(ticketId, null),
                    null, TicketWithWeight.class);

            if (null == ticketWithWeight) {
                ticketWithWeight = new TicketWithWeight(new BigDecimal(0));
            }

            // Refresh ticket
            ticketWithWeight.setTicketId(ticketId);
            ticketWithWeight.setDomainCode(ticket.getDomainCode());

            // Setting count
            long count = userStatisticsService.countViewOfToday(uid, ticketId);
            ticketWithWeight.setCount(count);

            rst.add(ticketWithWeight);
        }

        Collections.sort(rst);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Sorted: {}", rst);
        }

        TicketWithWeight finalTicket = getNext(rst, last);
        for (Ticket ticket : tickets) {
            if (ticket.getId().equals(finalTicket.getTicketId())) {
                return ticket;
            }
        }
        return null;
    }

    private static TicketWithWeight getNext(List<TicketWithWeight> tickets, Ticket last) {
        TicketWithWeight next = tickets.get(0);
        if (last == null) {
            return next;
        }
        // 如果last=null，或者没有与last相同分类的对象，那么直接返回列表第一个元素。
        if (! existsWithTag(tickets, last.getDomainCode())) {
            return next;
        }

        // 如果类型相同，那么需要获取下一个不同类型的元素
        if (StringUtils.equals(last.getDomainCode(), next.getDomainCode())) {
            TicketWithWeight firstDifferentWithTag = getFirstDifferentWithTag(tickets, last.getDomainCode());
            if (null != firstDifferentWithTag) {
                return firstDifferentWithTag;
            }
        }
        return next;
    }

    private static boolean existsWithTag(List<TicketWithWeight> tickets, String domainOfLast) {
        for (TicketWithWeight ticket : tickets) {
            if (StringUtils.equals(ticket.getDomainCode(), domainOfLast)) {
                return true;
            }
        }
        return false;
    }

    private static TicketWithWeight getFirstDifferentWithTag(List<TicketWithWeight> tickets, String domainOfLast) {
        for (TicketWithWeight ticket : tickets) {
            if (!StringUtils.equals(ticket.getDomainCode(), domainOfLast)) {
                return ticket;
            }
        }
        return null;
    }
}
