package com.bxm.adsprod.service.ticket.filter;

import com.bxm.adsprod.common.interceptor.AbstractInterceptor;
import com.bxm.adsprod.common.interceptor.Interceptor;
import com.bxm.adsprod.common.interceptor.Invocation;
import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.adsprod.facade.media.Position;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 广告位行业地域拦截器
 *
 * @author allen
 * @date 2017-12-06
 */
@Component
public class PositionDomainRegionInterceptor extends AbstractInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(PositionDomainRegionInterceptor.class);

    @Autowired(required = false)
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired(required = false)
    @Qualifier("jedisUpdater")
    private Updater updater;

    public PositionDomainRegionInterceptor() {
        super();
    }

    public PositionDomainRegionInterceptor(Interceptor next) {
        super(next);
    }

    @Override
    protected void doIntercept(Invocation invocation) {
        FilterRequestModel requestModel = (FilterRequestModel) invocation.getRequestModel();

        if (requestModel.isEmpty()) {
            return;
        }

        Position position = requestModel.getPosition();
        if (null == position) {
            throw new NullPointerException("media cannot be null");
        }
        String domainCode = position.getDomainCode();
        if (StringUtils.isBlank(domainCode)) {
            return;
        }
        String regionCode = requestModel.getIprst().getRegioncode();
        if (StringUtils.isBlank(regionCode)) {
            return;
        }

        Map<String, Object> param = Maps.newHashMap();
        param.put(CachePushableFields.POSITION_ID, position.getId());
        param.put(CachePushableFields.DOMAIN_CODE, domainCode);

        Set<String> rule = fetcher.fetch(keyGenerator(param), null, Set.class);
        if (CollectionUtils.isEmpty(rule)) {
            return;
        }

        List<Ticket> tickets = requestModel.getTickets();

        Iterator<Ticket> iterator = tickets.iterator();
        while (iterator.hasNext()) {
            Ticket ticket = iterator.next();

            if (rule.contains(regionCode)) {
                iterator.remove();

                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[PositionDomainRegionInterceptor] remove {}.", ticket.getId());
                }
            }
        }
    }

    public static KeyGenerator keyGenerator(Map<String, Object> parameters) {

        return new KeyGenerator() {
            @Override
            public String generateKey() {
                return KeyBuilder.build("AD", "FILTER", "POSITION_DOMAIN",
                        parameters.get(CachePushableFields.POSITION_ID),
                        parameters.get(CachePushableFields.DOMAIN_CODE));
            }
        };
    }
}
