package com.bxm.adsprod.service.ticket.filter;

import com.alibaba.fastjson.JSONArray;
import com.bxm.adsprod.common.interceptor.AbstractInterceptor;
import com.bxm.adsprod.common.interceptor.Invocation;
import com.bxm.adsprod.facade.media.Position;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.service.media.PositionIdResolver;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.autoconfigure.jedis.JedisConfiguration;
import com.bxm.warcar.cache.impls.redis.JedisFetcher;
import com.bxm.warcar.utils.KeyBuilder;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import redis.clients.jedis.JedisPool;

import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;

/**
 * <h3>广告位标签地域拦截器(老平台实现)</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/25
 */
@Configuration
@EnableConfigurationProperties({JedisConfiguration.class, OldConfiguration.class})
public class PositionTagRegionForOldInterceptor extends AbstractInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(PositionTagRegionForOldInterceptor.class);
    private static final int VALUE_SPLIT_SIZE = 2;
    @Autowired
    private JedisConfiguration configuration;
    @Autowired
    private OldConfiguration oldConfiguration;
    @Autowired
    @Qualifier("jedisFetcherForOld")
    private Fetcher fetcher;

    @Override
    protected void doIntercept(Invocation invocation) {
        FilterRequestModel requestModel = (FilterRequestModel) invocation.getRequestModel();

        if (requestModel.isEmpty()) {
            return;
        }

        String city = requestModel.getCity() + "市";

        String time = getTimeFromFetcher();

        if (StringUtils.isBlank(time)) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("Cloud not found 'time' value from fetcher.");
            }
            return;
        }

        Position position = requestModel.getPosition();
        String positionId = position.getId();
        String[] resolve = PositionIdResolver.DEFAULT.resolve(positionId);

        if (null == resolve || resolve.length != VALUE_SPLIT_SIZE) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("Illegal positionId: {}", position.getId());
            }
            return;
        }
        String appkey = resolve[0];
        String business = resolve[1];

        List<Ticket> tickets = requestModel.getTickets();
        List<BigInteger> appAreaBlack = getAppAreaBlack(time, KeyBuilder.build(appkey, business, city));
        if (CollectionUtils.isNotEmpty(appAreaBlack)) {
            loopRemove(tickets, appAreaBlack, true);
            return;
        }
        List<BigInteger> appBlack = getAppBlack(time, KeyBuilder.build(appkey, business));
        if (CollectionUtils.isNotEmpty(appBlack)) {
            loopRemove(tickets, appBlack, true);
            return;
        }
        List<BigInteger> appWhite = getAppWhite(time, KeyBuilder.build(appkey, business));
        if (CollectionUtils.isNotEmpty(appWhite)) {
            loopRemove(tickets, appWhite, false);
        }
    }

    private void loopRemove(List<Ticket> tickets, List<BigInteger> cache, boolean blacklist) {
        Iterator<Ticket> iterator = tickets.iterator();
        for (; iterator.hasNext(); ) {
            Ticket ticket = iterator.next();
            BigInteger ticketId = ticket.getId();
            boolean removing = (blacklist && cache.contains(ticketId)) || (!blacklist && !cache.contains(ticketId));
            if (removing) {
                iterator.remove();
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[PositionTagRegionForOldInterceptor] remove {}.", ticketId);
                }
            }
        }
    }

    private String getTimeFromFetcher() {
        return fetcher.fetch(new KeyGenerator() {
                @Override
                public String generateKey() {
                    return KeyBuilder.build("certificate", "real", "key");
                }
            }, null, String.class);
    }

    /**
     * @param time 时间戳
     * @param field appkey:business:city
     * @return [ticketid, ticketid]
     */
    private List<BigInteger> getAppAreaBlack(String time, String field) {
        String cache = fetcher.hfetch(new KeyGenerator() {
            @Override
            public String generateKey() {
                return KeyBuilder.build("app_area", "black", time);
            }
        }, field, null, String.class);
        if (StringUtils.isBlank(cache)) {
            return null;
        }
        return JSONArray.parseArray(cache, BigInteger.class);
    }

    /**
     * @param time 时间戳
     * @param field appkey:business
     * @return [ticketid, ticketid]
     */
    private List<BigInteger> getAppBlack(String time, String field) {
        String cache = fetcher.hfetch(new KeyGenerator() {
            @Override
            public String generateKey() {
                return KeyBuilder.build("app", "black", time);
            }
        }, field, null, String.class);
        if (StringUtils.isBlank(cache)) {
            return null;
        }
        return JSONArray.parseArray(cache, BigInteger.class);
    }

    /**
     * @param time 时间戳
     * @param field appkey:business
     * @return [ticketid, ticketid]
     */
    private List<BigInteger> getAppWhite(String time, String field) {
        String cache = fetcher.hfetch(new KeyGenerator() {
            @Override
            public String generateKey() {
                return KeyBuilder.build("app", "white", time);
            }
        }, field, null, String.class);
        if (StringUtils.isBlank(cache)) {
            return null;
        }
        return JSONArray.parseArray(cache, BigInteger.class);
    }

    @Bean
    public JedisPool jedisPoolForOld() {
        return new JedisPool(oldConfiguration, oldConfiguration.getHost(), oldConfiguration.getPort(), oldConfiguration.getTimeout(),
                oldConfiguration.getPassword(), oldConfiguration.getDatabase());
    }

    @Bean
    public JedisFetcher jedisFetcherForOld() {
        return new JedisFetcher(jedisPoolForOld());
    }
}
