package com.bxm.adsprod.service.ticket.filter;

import com.bxm.adsprod.common.interceptor.AbstractInterceptor;
import com.bxm.adsprod.common.interceptor.Interceptor;
import com.bxm.adsprod.common.interceptor.Invocation;
import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.adsprod.facade.media.Position;
import com.bxm.adsprod.facade.tag.Tag;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.ip.IP;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 广告位标签地域拦截器
 *
 * @author allen
 * @date 2017-12-06
 */
@Component
public class PositionTagRegionInterceptor extends AbstractInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(PositionTagRegionInterceptor.class);

    @Autowired(required = false)
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired(required = false)
    @Qualifier("jedisUpdater")
    private Updater updater;

    public PositionTagRegionInterceptor() {
        super();
    }

    public PositionTagRegionInterceptor(Interceptor next) {
        super(next);
    }

    @Override
    protected void doIntercept(Invocation invocation) {
        FilterRequestModel requestModel = (FilterRequestModel) invocation.getRequestModel();

        if (requestModel.isEmpty()) {
            return;
        }

        Position position = requestModel.getPosition();
        if (null == position) {
            throw new NullPointerException("media cannot be null");
        }

        String positionId = position.getId();
        IP iprst = requestModel.getIprst();
        String regionCode = iprst.getRegioncode();
        if (StringUtils.isBlank(regionCode)) {
            return;
        }

        List<Ticket> tickets = requestModel.getTickets();
        Iterator<Ticket> iterator = tickets.iterator();
        while (iterator.hasNext()) {
            Ticket ticket = iterator.next();
            BigInteger ticketId = ticket.getId();

            List<Tag> tags = ticket.getTags();
            if (CollectionUtils.isEmpty(tags)) {
                continue;
            }

            for (Tag tag : tags) {
                String code = tag.getCode();

                Map<String, Object> parameters = Maps.newHashMap();
                parameters.put(CachePushableFields.POSITION_ID, positionId);
                parameters.put(CachePushableFields.TAG_CODE, code);

                Set<String> filters = fetcher.fetch(keyGenerator(parameters),null, Set.class);
                if (CollectionUtils.isEmpty(filters)) {
                    continue;
                }

                if (filters.contains(regionCode)) {
                    iterator.remove();

                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("[PositionTagRegionInterceptor] remove {}.", ticketId);
                    }
                    break;
                }
            }
        }
    }

    public static KeyGenerator keyGenerator(Map<String, Object> parameters) {
        return new KeyGenerator() {
            @Override
            public String generateKey() {
                return KeyBuilder.build("AD", "FILTER", "POSITION_TAG",
                        parameters.get(CachePushableFields.POSITION_ID),
                        parameters.get(CachePushableFields.TAG_CODE));
            }
        };
    }

}
