package com.bxm.adsprod.service.ticket.filter;

import com.bxm.adsprod.common.interceptor.AbstractInterceptor;
import com.bxm.adsprod.common.interceptor.Interceptor;
import com.bxm.adsprod.common.interceptor.Invocation;
import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.warcar.cache.DataExtractor;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 广告券区域拦截器
 *
 * @author allen
 * @date 2017-12-04
 */
@Component
public class TicketRegionInterceptor extends AbstractInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketRegionInterceptor.class);

    @Autowired(required = false)
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired(required = false)
    @Qualifier("jedisUpdater")
    private Updater updater;

    public TicketRegionInterceptor() {
        super();
    }

    public TicketRegionInterceptor(Interceptor next) {
        super(next);
    }

    @Override
    protected void doIntercept(Invocation invocation) {
        FilterRequestModel requestModel = (FilterRequestModel) invocation.getRequestModel();

        if (requestModel.isEmpty()) {
            return;
        }

        String regionCode = requestModel.getIprst().getRegioncode();
        if (StringUtils.isBlank(regionCode)) {
            return;
        }

        List<Ticket> tickets = requestModel.getTickets();

        Iterator<Ticket> iterator = tickets.iterator();

        while (iterator.hasNext()) {
            Ticket ticket = iterator.next();
            final BigInteger ticketId = ticket.getId();
            Map<String, Object> parameters = Maps.newHashMap();
            parameters.put(CachePushableFields.TICKET_ID, ticketId);

            KeyGenerator keyGenerator = keyGenerator(parameters);
            DataExtractor dataExtractor = getDataExtractor();

            Set<String> available = fetcher.fetch(keyGenerator, dataExtractor, Set.class);
            if (CollectionUtils.isEmpty(available)) {
                continue;
            }

            if (! available.contains(regionCode)) {
                iterator.remove();

                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[TicketRegionInterceptor] remove {}.", ticketId);
                }
            }
        }
    }

    private DataExtractor getDataExtractor() {
        return new DataExtractor<Set<String>>() {

                @Override
                public Set<String> extract() {
                    return null;
                }
            };
    }

    public static KeyGenerator keyGenerator(final Map<String, Object> parameters) {
        return new KeyGenerator() {
            @Override
            public String generateKey() {
                Object ticketId = parameters.get(CachePushableFields.TICKET_ID);
                return KeyBuilder.build("AD", "FILTER", "REGION", ticketId);
            }
        };
    }
}
