package com.bxm.adsprod.service.ticket.filter;

import com.bxm.adsprod.common.eventbus.EventPark;
import com.bxm.adsprod.common.interceptor.AbstractInterceptor;
import com.bxm.adsprod.common.interceptor.Interceptor;
import com.bxm.adsprod.common.interceptor.Invocation;
import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketStatisticsService;
import com.bxm.adsprod.model.so.rules.TicketTimelineRuleSo;
import com.bxm.adsprod.service.ticket.event.DailyBudgetNotEnoughEvent;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * <h3>广告券时间段拦截器</h3>
 * <p>广告券可设定开放的时间段，且时间段支持限额</p>
 *
 * @author allen
 * @since V1.0.0 2017/12/06
 */
@Component
public class TicketTimelineInterceptor extends AbstractInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketTimelineInterceptor.class);

    @Autowired(required = false)
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;
    @Autowired(required = false)
    private TicketStatisticsService ticketStatisticsService;
    @Autowired
    private EventPark eventPark;

    public TicketTimelineInterceptor() {
        super();
    }

    public TicketTimelineInterceptor(Interceptor next) {
        super(next);
    }

    @Override
    protected void doIntercept(Invocation invocation) {

        FilterRequestModel requestModel = (FilterRequestModel) invocation.getRequestModel();

        if (requestModel.isEmpty()) {
            return;
        }

        List<Ticket> tickets = requestModel.getTickets();
        Iterator<Ticket> iterator = tickets.iterator();

        int currentHour = Calendar.getInstance().get(Calendar.HOUR_OF_DAY);
        while (iterator.hasNext()) {
            Ticket ticket = iterator.next();
            Map<String, Object> parameters = Maps.newHashMap();
            BigInteger ticketId = ticket.getId();
            parameters.put(CachePushableFields.TICKET_ID, ticketId);
            TicketTimelineRuleSo rule = fetcher.fetch(keyGenerator(parameters), null, TicketTimelineRuleSo.class);
            if (null == rule) {
                // 全天开放，判断是否超过日限额
                long budgetOfToday = ticketStatisticsService.getBudgetOfToday(ticketId);
                Integer ticketBudgetDaily = ticket.getBudgetDaily();
                if (budgetOfToday >= ticketBudgetDaily) {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("[TicketTimelineInterceptor] removed [{}], 已超出今日预算 {}/{}。", ticketId, budgetOfToday, ticketBudgetDaily);
                    }
                    iterator.remove();

                    // Post message to event park.
                    eventPark.post(new DailyBudgetNotEnoughEvent(this, ticket));
                }
                continue;
            }

            List<TicketTimelineRuleSo.Entry> entries = rule.getEntries();

            boolean inTimelines = false;
            for (TicketTimelineRuleSo.Entry entry : entries) {
                int startHour = entry.getStartHour();
                int endHour = entry.getEndHour();
                long max = entry.getLimit();
                boolean isNow = currentHour >= startHour && currentHour < endHour;
                if (isNow) {

                    inTimelines = true;
                    // 判断当前时间段的限额是否超出
                    long budgetOfTimeline = ticketStatisticsService.getBudgetOfTimeline(ticketId, startHour, endHour);
                    if (budgetOfTimeline >= max) {
                        if (LOGGER.isDebugEnabled()) {
                            LOGGER.debug("[TicketTimelineInterceptor] remove [{}], 当前时间段 [{}-{}] 已超出预算 {}/{}。", ticketId, startHour, endHour, budgetOfTimeline, max);
                        }
                        iterator.remove();
                    }
                    break;
                }
            }

            if (! inTimelines) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[TicketTimelineInterceptor] remove [{}], 不在投放时间段。", ticketId);
                }
                iterator.remove();
            }
        }
    }

    public static KeyGenerator keyGenerator(Map<String, Object> parameters) {
        return new KeyGenerator() {
            @Override
            public String generateKey() {
                return KeyBuilder.build("AD", "FILTER", "TIMELINE",
                        parameters.get(CachePushableFields.TICKET_ID));
            }
        };
    }
}
