package com.bxm.adsprod.service.ticket.filter;

import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.adsprod.model.so.rules.TicketTimesRuleSo;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.service.user.UserService;
import com.bxm.adsprod.service.user.UserStatisticsService;
import com.bxm.adsprod.common.interceptor.AbstractInterceptor;
import com.bxm.adsprod.common.interceptor.Interceptor;
import com.bxm.adsprod.common.interceptor.Invocation;
import com.bxm.adsprod.service.ticket.filter.times.TimesRuleProcessor;
import com.bxm.adsprod.service.ticket.filter.times.TimesRuleProcessorFactory;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * 广告券用户频次拦截器
 *
 * @see TicketTimesRuleSo
 * @author allen
 * @date 2017-12-06
 */
@Component
public class TicketTimesInterceptor extends AbstractInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketTimesInterceptor.class);

    @Autowired(required = false)
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired(required = false)
    @Qualifier("jedisUpdater")
    private Updater updater;

    @Autowired(required = false)
    private UserService userService;

    @Autowired(required = false)
    private UserStatisticsService userStatisticsService;

    public TicketTimesInterceptor() {
        super();
    }

    public TicketTimesInterceptor(Interceptor next) {
        super(next);
    }

    @Override
    protected void doIntercept(Invocation invocation) {

        FilterRequestModel requestModel = (FilterRequestModel) invocation.getRequestModel();

        if (requestModel.isEmpty()) {
            return;
        }

        String uid = requestModel.getUid();

        if (StringUtils.isBlank(uid)) {
            throw new NullPointerException("uid cannot be blank");
        }

        List<Ticket> tickets = requestModel.getTickets();
        Iterator<Ticket> iterator = tickets.iterator();
        while (iterator.hasNext()) {
            Ticket ticket = iterator.next();
            BigInteger ticketId = ticket.getId();
            Map<String, Object> parameters = Maps.newHashMap();
            parameters.put(CachePushableFields.TICKET_ID, ticketId);
            TicketTimesRuleSo rule = fetcher.fetch(keyGenerator(parameters), null, TicketTimesRuleSo.class);
            if (null == rule) {
                continue;
            }
            List<TicketTimesRuleSo.Entry> entries = rule.getEntries();
            for (TicketTimesRuleSo.Entry entry : entries) {
                TimesRuleProcessor processor = TimesRuleProcessorFactory.create(userStatisticsService, entry);
                if (processor.isOutOfTimes(uid, ticketId)) {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("[TicketTimesInterceptor] remove {}.", ticketId);
                    }
                    iterator.remove();

                    // this ticket has been removed, pass the loop for rules.
                    break;
                }
            }
        }
    }

    public static KeyGenerator keyGenerator(Map<String, Object> parameters) {
        return new KeyGenerator() {
            @Override
            public String generateKey() {
                return KeyBuilder.build("AD", "FILTER", "TIMES",
                        parameters.get(CachePushableFields.TICKET_ID));
            }
        };
    }
}
