package com.bxm.adsprod.service.ticket.filter;

import com.bxm.adsprod.common.interceptor.AbstractInterceptor;
import com.bxm.adsprod.common.interceptor.Interceptor;
import com.bxm.adsprod.common.interceptor.Invocation;
import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketOnoff;
import com.bxm.adsprod.facade.user.UserGrade;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.Constants;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * <h3>用户评分拦截器</h3>
 * <p></p>
 *
 * @author allen
 * @since V1.0.0 2017/12/06
 */
@Component
public class TicketUserGradeInterceptor extends AbstractInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketUserGradeInterceptor.class);

    @Autowired
    @Qualifier("jedis1Fetcher")
    private Fetcher fetcher;

    public TicketUserGradeInterceptor() {
        super();
    }

    public TicketUserGradeInterceptor(Interceptor next) {
        super(next);
    }

    @Override
    protected void doIntercept(Invocation invocation) {
        FilterRequestModel requestModel = (FilterRequestModel) invocation.getRequestModel();

        if (requestModel.isEmpty()) {
            return;
        }

        // 获取用户评分信息
        UserGrade userGrade = null;
        if (! requestModel.isNoImei()) {
            String imei = requestModel.getImei();

            Map<String, Object> parameters = Maps.newHashMap();
            parameters.put(CachePushableFields.IMEI, imei);
            userGrade = fetcher.fetch(keyGenerator(parameters), null, UserGrade.class);
        }

        List<Ticket> tickets = requestModel.getTickets();
        Iterator<Ticket> iterator = tickets.iterator();
        while (iterator.hasNext()) {
            Ticket ticket = iterator.next();
            BigInteger ticketId = ticket.getId();

            List<TicketOnoff> onoffs = ticket.getOnoffs();
            if (CollectionUtils.isEmpty(onoffs)) {
                // 不限人群定向
                continue;
            }

            if (requestModel.isNoImei()) {
                // 无IMEI模式
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("无IMEI模式");
                }
                if (removeForPercent(iterator, onoffs, TicketOnoff.TYPE_NOIMEI)) {
                    debugRemoved(ticketId);
                }
                continue;
            }

            if (null == userGrade || ! userGrade.isValid()) {
                // 无效IMEI模式
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("无效IMEI模式");
                }
                if (removeForPercent(iterator, onoffs, TicketOnoff.TYPE_INVALIDIMEI)) {
                    debugRemoved(ticketId);
                }
                continue;
            }

            Map<String, UserGrade.Grade> grades = userGrade.getGrades();

            if (MapUtils.isEmpty(grades)) {
                // 没有用户评分，正常情况不会出现，上面isValid()已经判断过了。
                continue;
            }

            UserGrade.Grade grade = grades.get(ticketId.toString());
            if (null == grade) {
                // 该广告券没有获得分数，正常情况不会出现
                continue;
            }

            Byte isVeto = grade.getIsVeto();
            Double score = grade.getScore();
            double resetScore = resetScore(isVeto, score);
            if (resetScore <= 0) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[TicketUserGradeInterceptor] 广告券[{}]用户[{}]画像评分[{}]被否决。", ticketId, userGrade.getImei(), grade);
                }
                iterator.remove();
                continue;
            }

            Integer passScore = ticket.getUserGradePassScore();
            if (null == passScore) {
                continue;
            }

            if (resetScore < passScore) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[TicketUserGradeInterceptor] 广告券[{}]用户[{}]画像[{}]得分{}<{}不及格。", ticketId, userGrade.getImei(), grade, resetScore, passScore);
                }
                iterator.remove();
            }
        }
    }

    private void debugRemoved(BigInteger ticketId) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[TicketUserGradeInterceptor] remove {}.", ticketId);
        }
    }

    /**
     * 按开关配置的百分比命中率来决定是否需要删除这张广告券
     *
     * @param iterator 广告券列表
     * @param onoffs 当前广告券开关配置
     * @param type 开关类型
     * @return 是否已经过滤
     * @see TicketOnoff#TYPE_INVALIDIMEI
     * @see TicketOnoff#TYPE_NOIMEI
     *
     */
    private boolean removeForPercent(Iterator<Ticket> iterator, List<TicketOnoff> onoffs, int type) {
        TicketOnoff conf = getFirstItemForType(onoffs, type);

        if (null == conf) {
            // 未获取到配置
            iterator.remove();
            return true;
        }

        // 投放百分比
        String value = conf.getValue();
        int percent = NumberUtils.toInt(value, 0);
        int random = random();

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("命中率：{}%，命中结果：{}<={}", percent, random, percent);
        }
        if (percent == 0 || random > percent) {
            iterator.remove();
            return true;
        }

        return false;
    }

    /**
     * 随机一个的数字
     * @return 1-100
     */
    private int random() {
        return RandomUtils.nextInt(100) + 1;
    }

    private TicketOnoff getFirstItemForType(List<TicketOnoff> onoffs, int type) {
        for (TicketOnoff onoff : onoffs) {
            Integer t = onoff.getType();
            if (null != t && type == t) {
                return onoff;
            }
        }
        return null;
    }

    /**
     * 重设分数
     * <p>如果分数为-1，则直接将分数设置为0，否则执行分数重设后设置为得出的结果；</p>
     *
     * @param isVeto 1=否决、0=不否决
     * @param score 分数
     * @see Constants#YES
     * @see Constants#NO
     * @return
     *
     * <p>分数判断</p>
     * <li>分数≤0视为需要过滤的项目；</li>
     * <li>分数＝0需要继续判断运营配置可发状态及概率；</li>
     * <li>分数＞0则需要继续比对配置的分数线。</li>
     */
    private double resetScore(Byte isVeto, Double score) {
        if (null == isVeto || null == score) {
            return 0;
        }
        if (score == -1) {
            return 0;
        }
        else {
            return score * (isVeto == Constants.YES ? -1 : 1);
        }
    }

    public static KeyGenerator keyGenerator(Map<String, Object> parameters) {
        return new KeyGenerator() {
            @Override
            public String generateKey() {
                return KeyBuilder.build("AD", "USER",
                        parameters.get(CachePushableFields.IMEI));
            }
        };
    }

}
