package com.bxm.adsprod.service.ticket.pushable;

import com.bxm.adsprod.common.pushable.annotation.CachePush;
import com.bxm.adsprod.convert.ticket.TicketConvert;
import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.adsprod.facade.rules.Rule;
import com.bxm.adsprod.facade.ticket.TicketKeyGenerator;
import com.bxm.adsprod.facade.ticket.TicketOfRules;
import com.bxm.adsprod.facade.ticket.TicketPrice;
import com.bxm.adsprod.facade.ticket.TicketWeightMif;
import com.bxm.adsprod.model.so.rules.TicketPositionRuleSo;
import com.bxm.adsprod.model.so.rules.TicketRegionRuleSo;
import com.bxm.adsprod.model.so.rules.TicketTimelineRuleSo;
import com.bxm.adsprod.model.so.rules.TicketTimesRuleSo;
import com.bxm.adsprod.service.ticket.filter.TicketPositionInterceptor;
import com.bxm.adsprod.service.ticket.filter.TicketRegionInterceptor;
import com.bxm.adsprod.service.ticket.filter.TicketTimelineInterceptor;
import com.bxm.adsprod.service.ticket.filter.TicketTimesInterceptor;
import com.bxm.warcar.cache.DataExtractor;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.*;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <h3>广告券缓存推送</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/12
 */
@Component
@CachePush("TICKET")
public class TicketPushable extends JSONObjectPushable<TicketOfRules> implements HashUpdating<TicketOfRules> {

    private static final Logger LOGGER = LoggerFactory.getLogger(TicketPushable.class);

    @Autowired
    @Qualifier("jedisUpdater")
    private Updater updater;
    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired @Qualifier("ticketRegionPushable")
    private Pushable ticketRegionPushable;
    @Autowired @Qualifier("ticketPositionPushable")
    private Pushable ticketPositionPushable;
    @Autowired @Qualifier("ticketTimeLinePushable")
    private Pushable ticketTimeLinePushable;
    @Autowired @Qualifier("ticketTimesPushable")
    private Pushable ticketTimesPushable;
    @Autowired @Qualifier("ticketMifPushable")
    private Pushable ticketMifPushable;
    @Autowired @Qualifier("ticketPricePushable")
    private Pushable ticketPricePushable;


    public TicketPushable() {
        super(new Converter<TicketOfRules>() {
            @Override
            public Object convert(TicketOfRules object) {
                return TicketConvert.of(object);
            }
        });
    }

    @Override
    protected Class<TicketOfRules> getClsType() {
        return TicketOfRules.class;
    }

    @Override
    public Updater getUpdater() {
        return updater;
    }

    @Override
    public String getField(Map<String, Object> parameters, TicketOfRules object) {
        return String.valueOf(object.getId());
    }

    @Override
    protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
        return TicketKeyGenerator.getAllTickets();
    }

    @Override
    protected void beforeUpdate(Map<String, Object> parameters, byte[] data, TicketOfRules object) {
        updater.hremove(getKeyGenerator(parameters), getField(parameters, object));

        Rule region = object.getRegion();
        ticketRegionPushable.push(parameters, null == region ? null : JsonHelper.convert2bytes(region));

        Rule position = object.getPosition();
        ticketPositionPushable.push(parameters, null == position ? null : JsonHelper.convert2bytes(position));

        Rule timeline = object.getTimeline();
        ticketTimeLinePushable.push(parameters, null == timeline ? null : JsonHelper.convert2bytes(timeline));

        Rule times = object.getTimes();
        ticketTimesPushable.push(parameters, null == times ? null : JsonHelper.convert2bytes(times));

        List<TicketWeightMif> mifs = object.getMifs();
        ticketMifPushable.push(parameters, null == mifs ? null : JsonHelper.convert2bytes(mifs));

        List<TicketPrice> prices = object.getPrices();
        ticketPricePushable.push(parameters, null == prices ? null : JsonHelper.convert2bytes(prices));
    }

    @Bean
    public Pushable ticketRegionPushable() {
        return new JSONObjectPushable<TicketRegionRuleSo>(new Converter<TicketRegionRuleSo>() {
            @Override
            public Object convert(TicketRegionRuleSo object) {
                if (null == object) {
                    return null;
                }
                Set<String> result = Sets.newHashSet();
                List<String> entries = object.getEntries();
                result.addAll(entries);
                return result;
            }
        }) {

            @Override
            protected Class<TicketRegionRuleSo> getClsType() {
                return TicketRegionRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                return TicketRegionInterceptor.keyGenerator(parameters);
            }
        };
    }

    @Bean
    public Pushable ticketTimeLinePushable() {
        return new JSONObjectPushable<TicketTimelineRuleSo>() {

            @Override
            protected Class<TicketTimelineRuleSo> getClsType() {
                return TicketTimelineRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                return TicketTimelineInterceptor.keyGenerator(parameters);
            }
        };
    }

    @Bean
    public Pushable ticketTimesPushable() {
        return new JSONObjectPushable<TicketTimesRuleSo>() {

            @Override
            protected Class<TicketTimesRuleSo> getClsType() {
                return TicketTimesRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                return TicketTimesInterceptor.keyGenerator(parameters);
            }
        };
    }

    @Bean
    public Pushable ticketPositionPushable() {
        return new JSONObjectPushable<TicketPositionRuleSo>() {

            @Override
            protected Class<TicketPositionRuleSo> getClsType() {
                return TicketPositionRuleSo.class;
            }

            @Override
            public Updater getUpdater() {
                return updater;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters) {
                return TicketPositionInterceptor.keyGenerator(parameters);
            }
        };
    }

    @Bean
    public Pushable ticketMifPushable() {
        return new JSONArrayPushable<TicketWeightMif>(new Converter<TicketWeightMif>() {
            @Override
            public Object convert(TicketWeightMif object) {
                return new BigDecimal(object.getMif()).divide(new BigDecimal(100), 7, RoundingMode.HALF_UP).doubleValue();
            }
        }) {

            @Override
            protected Class<TicketWeightMif> getClsType() {
                return TicketWeightMif.class;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters, TicketWeightMif object) {
                parameters.put(CachePushableFields.POSITION_ID, object.getPositionId());
                return TicketKeyGenerator.getTicketMif(parameters);
            }

            @Override
            protected Updater getUpdater() {
                return updater;
            }
        };
    }

    @Bean
    public Pushable ticketPricePushable() {
        return new JSONArrayPushable<TicketPrice>() {

            @Override
            protected Class<TicketPrice> getClsType() {
                return TicketPrice.class;
            }

            @Override
            protected KeyGenerator getKeyGenerator(Map<String, Object> parameters, TicketPrice object) {
                parameters.put(CachePushableFields.POSITION_ID, object.getPositionId());
                return TicketKeyGenerator.getTicketPrice(parameters);
            }

            @Override
            protected Updater getUpdater() {
                return updater;
            }

            @Override
            protected void afterUpdated(Map<String, Object> parameters, TicketPrice object) {
                if (null == object) {
                    return;
                }
                if (null == object.getBeforePrice() || null == object.getAfterPrice()) {
                    if (LOGGER.isWarnEnabled()) {
                        LOGGER.warn("Before/After price cannot be null. {}", object);
                    }
                    return;
                }
                // 价格浮动比（%）=调整价/原单价
                // 查看今天是否已经保存过价格，如果没有那么取这次保存前的价格=原单价
                parameters.put(CachePushableFields.POSITION_ID, object.getPositionId());
                Integer first = getFirst(parameters, object);

                BigDecimal floatPercent = new BigDecimal(100);
                if (null != first && first != 0) {
                    // 设置浮动比，有效期1小时
                    floatPercent = new BigDecimal(object.getAfterPrice()).divide(new BigDecimal(first), 7, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
                }

                int oneHour = 60 * 60;
                updater.update(TicketKeyGenerator.getTicketPriceFloatPercent(parameters), floatPercent, oneHour);

                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Setting price float percent：parameters={}, first price={}, float percent={}.", parameters, first, floatPercent);
                }
            }

            private Integer getFirst(Map<String, Object> parameters, TicketPrice object) {
                return fetcher.fetch(TicketKeyGenerator.getTicketPriceOfTodayFirst(parameters),
                                    new DataExtractor<Integer>() {
                                        @Override
                                        public Integer extract() {
                                            Integer beforePrice = object.getBeforePrice();
                                            return null == beforePrice || beforePrice == 0 ? null : beforePrice;
                                        }
                                    }, Integer.class, TypeHelper.castToInt(DateHelper.getRemainSecondsOfToday()));
            }
        };
    }
}
