package com.bxm.adx.common.buy.buyer;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.buy.dsp.Dsp;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.util.Map;
import java.util.Objects;

/**
 * 抽象的 Http 协议买家实现。
 * @author allen
 * @since 2019-12-13
 */
@Slf4j
public abstract class AbstractHttpBuyer extends AbstractBuyer {

    private final HttpClient httpClient;

    public AbstractHttpBuyer(AdxProperties properties) {
        int timeout = properties.getBuyerBiddingTimeOutInMillis() / 2;
        RequestConfig config = RequestConfig.custom().setConnectTimeout(timeout).setSocketTimeout(timeout).build();
        httpClient = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        AbstractHttpBuyer that = (AbstractHttpBuyer) o;
        String code = getCode();
        return Objects.equals(code, that.getCode());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getCode());
    }

    /**
     * 返回请求地址
     * @return URL
     */
    protected String getUri() {
        Dsp dsp = getDsp();
        return dsp.getApiUrl();
    }

    /**
     * 返回请求Header
     * @return Headers
     */
    protected Map<String, String> getHeaders() {
        return null;
    }

    /**
     * 返回请求方式
     * @return 请求方式
     */
    protected Method getMethod() {
        return Method.POST;
    }

    @Override
    protected byte[] request(byte[] requestBody) {
        try {
            String uri = getUri();
            if (StringUtils.isBlank(uri)) {
                throw new NullPointerException("uri");
            }
            Map<String, String> headers = getHeaders();
            HttpUriRequest request =
                    getMethod() == Method.GET
                    ? buildGet(uri, headers, RequestBodyCodec.convert(requestBody))
                    : buildPost(uri, headers, requestBody);
            HttpResponse httpResponse = httpClient.execute(request);
            return EntityUtils.toByteArray(httpResponse.getEntity());
        } catch (IOException e) {
            log.warn("request: {}", e.getMessage());
        } catch (NullPointerException e) {
            log.warn("request: ", e);
        }
        return null;
    }

    private HttpGet buildGet(String uri, Map<String, String> headers, Map<String, Object> params) {
        HttpGet get = new HttpGet(appendParameterToUrl(params, uri));
        if (MapUtils.isNotEmpty(headers)) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                get.addHeader(entry.getKey(), entry.getValue());
            }
        }

        return get;
    }

    private HttpPost buildPost(String uri, Map<String, String> headers, byte[] requestBody) {
        HttpPost post = new HttpPost(uri);
        post.setEntity(new ByteArrayEntity(requestBody));
        if (MapUtils.isNotEmpty(headers)) {
            for (Map.Entry<String, String> entry : headers.entrySet()) {
                post.addHeader(entry.getKey(), entry.getValue());
            }
        }
        return post;
    }

    protected String appendParameterToUrl(Map<String, Object> params, String url) {
        StringBuilder uri = new StringBuilder(255);
        if (!StringUtils.endsWith(url, "?")) {
            uri.append("?");
        }
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();
            if (StringUtils.isNotBlank(key) && Objects.nonNull(value)) {
                uri.append(key).append("=").append(value).append("&");
            }
        }
        url += StringUtils.removeEnd(uri.toString(), "&");
        return url;
    }

    public enum Method {
        /**
         * Get method
         */
        GET,
        /**
         * Post method
         */
        POST
    }
}
