package com.bxm.adx.common.market.exchange;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.MarketRequest;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.context.annotation.Configuration;

import java.util.List;
import java.util.Set;
import java.util.concurrent.*;

/**
 * @author allen
 * @since 2019-12-18
 */
@Slf4j
@Configuration
public class DefaultPriorityExchanger implements PriorityExchanger {

    private final AdxProperties properties;
    private final Exchanger exchanger;

    public DefaultPriorityExchanger(AdxProperties properties, Exchanger exchanger) {
        this.properties = properties;
        this.exchanger = exchanger;
    }

    @Override
    public List<Deal> exchange(MarketOrders orders) {
        BidRequest request = orders.getBidRequest();
        List<MarketRequest> requestList = orders.getRequests();
        if (CollectionUtils.isEmpty(requestList)) {
            return null;
        }
        int size = requestList.size();
        final ThreadPoolExecutor threadPoolExecutor = new ThreadPoolExecutor(size, size, 5, TimeUnit.SECONDS,
                new LinkedBlockingQueue<>(), new NamedThreadFactory("priority"));
        threadPoolExecutor.allowCoreThreadTimeOut(true);
        List<Future<List<Deal>>> futures = Lists.newArrayListWithCapacity(size);
        for (MarketRequest marketRequest : requestList) {
            Set<Buyer> buyers = marketRequest.getBuyers();
            Set<Position> positions = marketRequest.getPositions();
            Set<String> positionIds = Sets.newHashSet();
            for (Position position : positions) {
                positionIds.add(position.getPositionId());
            }
            List<Impression> imps = Lists.newArrayList(request.getImps());

            imps.removeIf(impression -> !positionIds.contains(impression.getTag_id()));
            final BidRequest e = new BidRequest();
            BeanUtils.copyProperties(request, e);
            e.setImps(imps);

            Future<List<Deal>> future = threadPoolExecutor.submit(() -> exchanger.bidding(e, buyers));
            futures.add(future);
        }
        List<Deal> result = Lists.newArrayList();
        for (Future<List<Deal>> future : futures) {
            try {
                List<Deal> deals = future.get();
//                List<Deal> deals = future.get(properties.getBuyerConcurrentBiddingTimeOutInMillis(), TimeUnit.MILLISECONDS);
                if (CollectionUtils.isNotEmpty(deals)) {
                    result.addAll(deals);
                }
            } catch (InterruptedException | ExecutionException e) {
                // 超时或意外终止
                future.cancel(true);
                log.warn("execute: {}", e.getMessage());
            }
        }
//        threadPoolExecutor.shutdownNow();
        return result;
    }
}
